<?php
session_start();
require_once 'config.php';

// Check if user is logged in and is admin
if (!isset($_SESSION['user_id']) || $_SESSION['role'] !== 'admin') {
    header('Location: index.php');
    exit;
}

$page_title = 'Active Sessions';
$conn = getDBConnection();

// Handle actions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';
    
    if ($action === 'kill_session') {
        $session_id = $_POST['session_id'] ?? '';
        
        $stmt = $conn->prepare("DELETE FROM user_sessions WHERE session_id = ?");
        $stmt->bind_param("s", $session_id);
        $stmt->execute();
        $stmt->close();
        
        $success = "Session terminated successfully";
    }
    elseif ($action === 'cleanup_expired') {
        $conn->query("DELETE FROM user_sessions WHERE expires_at < NOW()");
        $success = "Expired sessions cleaned up";
    }
}

// Get active sessions
$sessions = $conn->query("
    SELECT 
        s.id,
        s.session_id,
        s.ip_address,
        s.user_agent,
        s.created_date,
        s.last_activity,
        s.expires_at,
        u.username,
        u.email,
        u.full_name
    FROM user_sessions s
    JOIN users u ON s.user_id = u.id
    WHERE s.expires_at > NOW()
    ORDER BY s.last_activity DESC
");

// Count stats
$stats = [];
$stats['active_sessions'] = $sessions->num_rows;
$stats['expired_sessions'] = $conn->query("SELECT COUNT(*) as count FROM user_sessions WHERE expires_at < NOW()")->fetch_assoc()['count'];
$stats['unique_users'] = $conn->query("SELECT COUNT(DISTINCT user_id) as count FROM user_sessions WHERE expires_at > NOW()")->fetch_assoc()['count'];

include 'includes/header.php';
?>

<style>
    .stat-card {
        background: white;
        border-radius: 12px;
        padding: 20px;
        box-shadow: 0 2px 8px rgba(0,0,0,0.1);
        margin-bottom: 20px;
    }
    
    .stat-number {
        font-size: 2.5rem;
        font-weight: 700;
        margin: 10px 0;
    }
    
    .table-container {
        background: white;
        border-radius: 12px;
        padding: 20px;
        box-shadow: 0 2px 8px rgba(0,0,0,0.1);
        margin-bottom: 30px;
    }
    
    .session-active {
        display: inline-block;
        width: 10px;
        height: 10px;
        background: #10b981;
        border-radius: 50%;
        animation: pulse 2s infinite;
    }
    
    @keyframes pulse {
        0%, 100% {
            opacity: 1;
        }
        50% {
            opacity: 0.5;
        }
    }
    
    .user-agent {
        font-size: 12px;
        color: #6b7280;
        font-family: monospace;
    }
</style>

<?php if (isset($success)): ?>
<div class="alert alert-success">
    <i class="fas fa-check-circle me-2"></i>
    <?php echo htmlspecialchars($success); ?>
</div>
<?php endif; ?>

<div class="page-header">
    <h1 class="page-title">
        <i class="fas fa-wifi"></i>
        Active Sessions
    </h1>
    <p class="text-muted">Monitor and manage active user sessions</p>
</div>

<!-- Statistics -->
<div class="row">
    <div class="col-md-4">
        <div class="stat-card text-center">
            <i class="fas fa-circle-notch fa-spin fa-2x" style="color: #10b981;"></i>
            <div class="stat-number" style="color: #10b981;"><?php echo $stats['active_sessions']; ?></div>
            <div>Active Sessions</div>
        </div>
    </div>
    <div class="col-md-4">
        <div class="stat-card text-center">
            <i class="fas fa-users fa-2x" style="color: #667eea;"></i>
            <div class="stat-number" style="color: #667eea;"><?php echo $stats['unique_users']; ?></div>
            <div>Online Users</div>
        </div>
    </div>
    <div class="col-md-4">
        <div class="stat-card text-center">
            <i class="fas fa-clock fa-2x" style="color: #f59e0b;"></i>
            <div class="stat-number" style="color: #f59e0b;"><?php echo $stats['expired_sessions']; ?></div>
            <div>Expired Sessions</div>
        </div>
    </div>
</div>

<!-- Actions -->
<div class="mb-3">
    <form method="POST" action="" style="display:inline;">
        <input type="hidden" name="action" value="cleanup_expired">
        <button type="submit" class="btn btn-warning">
            <i class="fas fa-broom me-2"></i>
            Cleanup Expired Sessions
        </button>
    </form>
    
    <a href="admin_panel.php" class="btn btn-secondary">
        <i class="fas fa-arrow-left me-2"></i>
        Back to Admin Panel
    </a>
</div>

<!-- Sessions Table -->
<div class="table-container">
    <h4 class="mb-4">
        <i class="fas fa-list me-2"></i>
        Currently Active Sessions
    </h4>
    
    <?php if ($sessions->num_rows == 0): ?>
        <div class="alert alert-info">
            <i class="fas fa-info-circle me-2"></i>
            No active sessions found.
        </div>
    <?php else: ?>
    <div class="table-responsive">
        <table class="table table-hover">
            <thead>
                <tr>
                    <th>Status</th>
                    <th>User</th>
                    <th>Email</th>
                    <th>IP Address</th>
                    <th>Browser/Device</th>
                    <th>Login Time</th>
                    <th>Last Activity</th>
                    <th>Expires</th>
                    <th>Actions</th>
                </tr>
            </thead>
            <tbody>
                <?php while ($session = $sessions->fetch_assoc()): 
                    $is_current = ($session['session_id'] === session_id());
                    $minutes_idle = round((time() - strtotime($session['last_activity'])) / 60);
                    $minutes_until_expire = round((strtotime($session['expires_at']) - time()) / 60);
                ?>
                <tr <?php echo $is_current ? 'style="background: #f0fdf4;"' : ''; ?>>
                    <td>
                        <span class="session-active" title="Active"></span>
                    </td>
                    <td>
                        <strong><?php echo htmlspecialchars($session['username']); ?></strong>
                        <?php if ($is_current): ?>
                            <span class="badge bg-success ms-2">YOU</span>
                        <?php endif; ?>
                    </td>
                    <td><?php echo htmlspecialchars($session['email']); ?></td>
                    <td><code><?php echo htmlspecialchars($session['ip_address']); ?></code></td>
                    <td>
                        <div class="user-agent" title="<?php echo htmlspecialchars($session['user_agent']); ?>">
                            <?php 
                            $ua = $session['user_agent'];
                            if (strpos($ua, 'Mobile') !== false) {
                                echo '<i class="fas fa-mobile-alt me-1"></i> Mobile';
                            } elseif (strpos($ua, 'Chrome') !== false) {
                                echo '<i class="fab fa-chrome me-1"></i> Chrome';
                            } elseif (strpos($ua, 'Firefox') !== false) {
                                echo '<i class="fab fa-firefox me-1"></i> Firefox';
                            } elseif (strpos($ua, 'Safari') !== false) {
                                echo '<i class="fab fa-safari me-1"></i> Safari';
                            } else {
                                echo '<i class="fas fa-globe me-1"></i> Browser';
                            }
                            ?>
                        </div>
                    </td>
                    <td>
                        <?php echo date('M j, g:i A', strtotime($session['created_date'])); ?>
                    </td>
                    <td>
                        <?php echo date('M j, g:i A', strtotime($session['last_activity'])); ?>
                        <br>
                        <small class="text-muted">(<?php echo $minutes_idle; ?> min ago)</small>
                    </td>
                    <td>
                        <?php if ($minutes_until_expire < 5): ?>
                            <span class="text-danger">
                                <i class="fas fa-exclamation-triangle me-1"></i>
                                <?php echo $minutes_until_expire; ?> min
                            </span>
                        <?php else: ?>
                            <?php echo $minutes_until_expire; ?> min
                        <?php endif; ?>
                    </td>
                    <td>
                        <?php if (!$is_current): ?>
                        <form method="POST" style="display:inline;" onsubmit="return confirm('Terminate this session? The user will be logged out immediately.');">
                            <input type="hidden" name="action" value="kill_session">
                            <input type="hidden" name="session_id" value="<?php echo htmlspecialchars($session['session_id']); ?>">
                            <button type="submit" class="btn btn-sm btn-danger" title="Terminate Session">
                                <i class="fas fa-times-circle"></i> Kill
                            </button>
                        </form>
                        <?php else: ?>
                            <span class="text-muted">Current</span>
                        <?php endif; ?>
                    </td>
                </tr>
                <?php endwhile; ?>
            </tbody>
        </table>
    </div>
    <?php endif; ?>
</div>

<!-- Legend -->
<div class="alert alert-info">
    <h6><i class="fas fa-info-circle me-2"></i>Session Information:</h6>
    <ul class="mb-0">
        <li><strong>One Device Per License:</strong> Each user can only be logged in on one device at a time</li>
        <li><strong>Session Timeout:</strong> Sessions expire after 30 minutes of inactivity</li>
        <li><strong>Force Login:</strong> When a user logs in on a new device, their old session is terminated</li>
        <li><strong>Auto Cleanup:</strong> Expired sessions are automatically removed</li>
    </ul>
</div>

<?php
$conn->close();
include 'includes/footer.php';
?>