<?php
session_start();
require_once 'config.php';

// Check if user is logged in and is admin
if (!isset($_SESSION['user_id']) || $_SESSION['role'] !== 'admin') {
    header('Location: index.php');
    exit;
}

$page_title = 'Admin Panel';
$conn = getDBConnection();

// Handle actions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';
    
    if ($action === 'generate_license') {
        $prefix = strtoupper($_POST['prefix'] ?? 'LIC');
        $validity_days = intval($_POST['validity_days'] ?? 365);
        $max_users = intval($_POST['max_users'] ?? 1);
        $notes = $_POST['notes'] ?? '';
        
        // Generate unique license key
        $license_key = $prefix . '-' . date('Y') . '-' . strtoupper(bin2hex(random_bytes(4))) . '-' . strtoupper(bin2hex(random_bytes(2)));
        $expires_date = $validity_days > 0 ? date('Y-m-d H:i:s', strtotime("+$validity_days days")) : NULL;
        
        $stmt = $conn->prepare("INSERT INTO license_keys (license_key, status, expires_date, max_users, notes) VALUES (?, 'available', ?, ?, ?)");
        $stmt->bind_param("ssis", $license_key, $expires_date, $max_users, $notes);
        $stmt->execute();
        $stmt->close();
        
        $success = "License key generated: $license_key";
    }
    elseif ($action === 'toggle_user_status') {
        $user_id = intval($_POST['user_id'] ?? 0);
        $new_status = $_POST['new_status'] ?? 'active';
        
        $stmt = $conn->prepare("UPDATE users SET status = ? WHERE id = ?");
        $stmt->bind_param("si", $new_status, $user_id);
        $stmt->execute();
        $stmt->close();
        
        $success = "User status updated";
    }
    elseif ($action === 'delete_user') {
        $user_id = intval($_POST['user_id'] ?? 0);
        
        // Don't allow deleting self
        if ($user_id != $_SESSION['user_id']) {
            $stmt = $conn->prepare("DELETE FROM users WHERE id = ?");
            $stmt->bind_param("i", $user_id);
            $stmt->execute();
            $stmt->close();
            
            $success = "User deleted";
        }
    }
}

// Get statistics
$stats = [];
$stats['total_users'] = $conn->query("SELECT COUNT(*) as count FROM users")->fetch_assoc()['count'];
$stats['active_users'] = $conn->query("SELECT COUNT(*) as count FROM users WHERE status='active'")->fetch_assoc()['count'];
$stats['total_licenses'] = $conn->query("SELECT COUNT(*) as count FROM license_keys")->fetch_assoc()['count'];
$stats['available_licenses'] = $conn->query("SELECT COUNT(*) as count FROM license_keys WHERE status='available'")->fetch_assoc()['count'];

// Get users
$users = $conn->query("SELECT id, username, email, full_name, license_key, status, role, created_date, last_login FROM users ORDER BY created_date DESC");

// Get license keys
$licenses = $conn->query("SELECT id, license_key, status, created_date, expires_date, used_date, max_users, notes FROM license_keys ORDER BY created_date DESC LIMIT 50");

include 'includes/header.php';
?>

<style>
    .stat-card {
        background: white;
        border-radius: 12px;
        padding: 20px;
        box-shadow: 0 2px 8px rgba(0,0,0,0.1);
        margin-bottom: 20px;
    }
    
    .stat-number {
        font-size: 2.5rem;
        font-weight: 700;
        margin: 10px 0;
    }
    
    .table-container {
        background: white;
        border-radius: 12px;
        padding: 20px;
        box-shadow: 0 2px 8px rgba(0,0,0,0.1);
        margin-bottom: 30px;
    }
    
    .badge-custom {
        padding: 5px 12px;
        border-radius: 20px;
        font-size: 12px;
        font-weight: 600;
    }
    
    .badge-active {
        background: #d1fae5;
        color: #065f46;
    }
    
    .badge-pending {
        background: #fef3c7;
        color: #92400e;
    }
    
    .badge-suspended {
        background: #fee2e2;
        color: #991b1b;
    }
    
    .badge-available {
        background: #dbeafe;
        color: #1e40af;
    }
    
    .badge-used {
        background: #e5e7eb;
        color: #374151;
    }
    
    .form-card {
        background: white;
        border-radius: 12px;
        padding: 25px;
        box-shadow: 0 2px 8px rgba(0,0,0,0.1);
        margin-bottom: 30px;
    }
</style>

<?php if (isset($success)): ?>
<div class="alert alert-success">
    <i class="fas fa-check-circle me-2"></i>
    <?php echo htmlspecialchars($success); ?>
</div>
<?php endif; ?>

<div class="page-header">
    <h1 class="page-title">
        <i class="fas fa-user-shield"></i>
        Admin Panel
    </h1>
    <p class="text-muted">Manage users and license keys</p>
</div>

<!-- Statistics -->
<div class="row">
    <div class="col-md-3">
        <div class="stat-card text-center">
            <i class="fas fa-users fa-2x" style="color: #667eea;"></i>
            <div class="stat-number" style="color: #667eea;"><?php echo $stats['total_users']; ?></div>
            <div>Total Users</div>
        </div>
    </div>
    <div class="col-md-3">
        <div class="stat-card text-center">
            <i class="fas fa-user-check fa-2x" style="color: #10b981;"></i>
            <div class="stat-number" style="color: #10b981;"><?php echo $stats['active_users']; ?></div>
            <div>Active Users</div>
        </div>
    </div>
    <div class="col-md-3">
        <div class="stat-card text-center">
            <i class="fas fa-key fa-2x" style="color: #f59e0b;"></i>
            <div class="stat-number" style="color: #f59e0b;"><?php echo $stats['total_licenses']; ?></div>
            <div>Total Licenses</div>
        </div>
    </div>
    <div class="col-md-3">
        <div class="stat-card text-center">
            <i class="fas fa-unlock fa-2x" style="color: #3b82f6;"></i>
            <div class="stat-number" style="color: #3b82f6;"><?php echo $stats['available_licenses']; ?></div>
            <div>Available Licenses</div>
        </div>
    </div>
</div>

<!-- Generate License Key Form -->
<div class="form-card">
    <h4 class="mb-4">
        <i class="fas fa-plus-circle me-2"></i>
        Generate New License Key
    </h4>
    <form method="POST" action="">
        <input type="hidden" name="action" value="generate_license">
        <div class="row">
            <div class="col-md-3">
                <label class="form-label">Prefix</label>
                <input type="text" name="prefix" class="form-control" value="PRO" maxlength="10" required>
            </div>
            <div class="col-md-3">
                <label class="form-label">Validity (Days)</label>
                <input type="number" name="validity_days" class="form-control" value="365" min="0">
                <small class="text-muted">0 = Lifetime</small>
            </div>
            <div class="col-md-2">
                <label class="form-label">Max Users</label>
                <input type="number" name="max_users" class="form-control" value="1" min="1" required>
            </div>
            <div class="col-md-4">
                <label class="form-label">Notes</label>
                <input type="text" name="notes" class="form-control" placeholder="Optional notes">
            </div>
        </div>
        <div class="mt-3">
            <button type="submit" class="btn btn-primary">
                <i class="fas fa-key me-2"></i>
                Generate License Key
            </button>
        </div>
    </form>
</div>

<!-- Users Table -->
<div class="table-container">
    <h4 class="mb-4">
        <i class="fas fa-users me-2"></i>
        Registered Users
    </h4>
    <div class="table-responsive">
        <table class="table table-hover">
            <thead>
                <tr>
                    <th>ID</th>
                    <th>Username</th>
                    <th>Email</th>
                    <th>Full Name</th>
                    <th>Role</th>
                    <th>Status</th>
                    <th>License Key</th>
                    <th>Registered</th>
                    <th>Last Login</th>
                    <th>Actions</th>
                </tr>
            </thead>
            <tbody>
                <?php while ($user = $users->fetch_assoc()): ?>
                <tr>
                    <td><?php echo $user['id']; ?></td>
                    <td><strong><?php echo htmlspecialchars($user['username']); ?></strong></td>
                    <td><?php echo htmlspecialchars($user['email']); ?></td>
                    <td><?php echo htmlspecialchars($user['full_name']); ?></td>
                    <td>
                        <?php if ($user['role'] === 'admin'): ?>
                            <span class="badge bg-danger">Admin</span>
                        <?php else: ?>
                            <span class="badge bg-secondary">User</span>
                        <?php endif; ?>
                    </td>
                    <td>
                        <span class="badge-custom badge-<?php echo $user['status']; ?>">
                            <?php echo ucfirst($user['status']); ?>
                        </span>
                    </td>
                    <td><code><?php echo htmlspecialchars($user['license_key']); ?></code></td>
                    <td><?php echo date('M j, Y', strtotime($user['created_date'])); ?></td>
                    <td><?php echo $user['last_login'] ? date('M j, g:i A', strtotime($user['last_login'])) : 'Never'; ?></td>
                    <td>
                        <?php if ($user['id'] != $_SESSION['user_id']): ?>
                        <form method="POST" style="display:inline;" onsubmit="return confirm('Are you sure?');">
                            <input type="hidden" name="action" value="toggle_user_status">
                            <input type="hidden" name="user_id" value="<?php echo $user['id']; ?>">
                            <?php if ($user['status'] === 'active'): ?>
                                <input type="hidden" name="new_status" value="suspended">
                                <button type="submit" class="btn btn-sm btn-warning" title="Suspend">
                                    <i class="fas fa-ban"></i>
                                </button>
                            <?php else: ?>
                                <input type="hidden" name="new_status" value="active">
                                <button type="submit" class="btn btn-sm btn-success" title="Activate">
                                    <i class="fas fa-check"></i>
                                </button>
                            <?php endif; ?>
                        </form>
                        <form method="POST" style="display:inline;" onsubmit="return confirm('Delete this user permanently?');">
                            <input type="hidden" name="action" value="delete_user">
                            <input type="hidden" name="user_id" value="<?php echo $user['id']; ?>">
                            <button type="submit" class="btn btn-sm btn-danger" title="Delete">
                                <i class="fas fa-trash"></i>
                            </button>
                        </form>
                        <?php endif; ?>
                    </td>
                </tr>
                <?php endwhile; ?>
            </tbody>
        </table>
    </div>
</div>

<!-- License Keys Table -->
<div class="table-container">
    <h4 class="mb-4">
        <i class="fas fa-key me-2"></i>
        License Keys (Last 50)
    </h4>
    <div class="table-responsive">
        <table class="table table-hover">
            <thead>
                <tr>
                    <th>ID</th>
                    <th>License Key</th>
                    <th>Status</th>
                    <th>Max Users</th>
                    <th>Created</th>
                    <th>Expires</th>
                    <th>Used Date</th>
                    <th>Notes</th>
                </tr>
            </thead>
            <tbody>
                <?php while ($license = $licenses->fetch_assoc()): ?>
                <tr>
                    <td><?php echo $license['id']; ?></td>
                    <td><code><?php echo htmlspecialchars($license['license_key']); ?></code></td>
                    <td>
                        <span class="badge-custom badge-<?php echo $license['status']; ?>">
                            <?php echo ucfirst($license['status']); ?>
                        </span>
                    </td>
                    <td><?php echo $license['max_users']; ?></td>
                    <td><?php echo date('M j, Y', strtotime($license['created_date'])); ?></td>
                    <td><?php echo $license['expires_date'] ? date('M j, Y', strtotime($license['expires_date'])) : 'Lifetime'; ?></td>
                    <td><?php echo $license['used_date'] ? date('M j, Y', strtotime($license['used_date'])) : '-'; ?></td>
                    <td><?php echo htmlspecialchars($license['notes'] ?? '-'); ?></td>
                </tr>
                <?php endwhile; ?>
            </tbody>
        </table>
    </div>
</div>

<?php
$conn->close();
include 'includes/footer.php';
?>