<?php
/**
 * Enhanced Authentication Check with Single Device Enforcement
 * Include this at the top of every protected page
 */

// Start session if not already started
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

// Check if user is logged in
if (!isset($_SESSION['user_id'])) {
    header('Location: login.php');
    exit;
}

// Get current session ID
$current_session_id = session_id();

// Check if this is still the active session in database
require_once __DIR__ . '/config.php';

try {
    $conn = getDBConnection();
    
    // Check if this session exists and is valid
    $stmt = $conn->prepare("SELECT session_id, expires_at FROM user_sessions WHERE user_id = ? AND session_id = ? LIMIT 1");
    $stmt->bind_param("is", $_SESSION['user_id'], $current_session_id);
    $stmt->execute();
    $result = $stmt->get_result();
    
    if ($result->num_rows === 0) {
        // Session was deleted (user logged in elsewhere)
        $stmt->close();
        $conn->close();
        
        session_unset();
        session_destroy();
        header('Location: login.php?kicked=1');
        exit;
    }
    
    $session_data = $result->fetch_assoc();
    $stmt->close();
    
    // Check if session has expired
    if (strtotime($session_data['expires_at']) < time()) {
        // Session expired
        $stmt2 = $conn->prepare("DELETE FROM user_sessions WHERE session_id = ?");
        $stmt2->bind_param("s", $current_session_id);
        $stmt2->execute();
        $stmt2->close();
        
        $conn->close();
        
        session_unset();
        session_destroy();
        header('Location: login.php?timeout=1');
        exit;
    }
    
    // Update session activity every 30 seconds to avoid too many DB writes
    if (!isset($_SESSION['last_db_update']) || (time() - $_SESSION['last_db_update']) > 30) {
        $new_expires = date('Y-m-d H:i:s', time() + 1800); // Extend 30 minutes
        $stmt2 = $conn->prepare("UPDATE user_sessions SET last_activity = NOW(), expires_at = ? WHERE session_id = ?");
        $stmt2->bind_param("ss", $new_expires, $current_session_id);
        $stmt2->execute();
        $stmt2->close();
        
        $_SESSION['last_db_update'] = time();
    }
    
    // Update last activity time in session
    $_SESSION['last_activity'] = time();
    
    // Optional: Verify user still exists and is active (check every 5 minutes)
    if (!isset($_SESSION['last_user_check']) || (time() - $_SESSION['last_user_check']) > 300) {
        $stmt3 = $conn->prepare("SELECT status FROM users WHERE id = ?");
        $stmt3->bind_param("i", $_SESSION['user_id']);
        $stmt3->execute();
        $user_result = $stmt3->get_result();
        
        if ($user_result->num_rows === 0) {
            // User no longer exists
            $stmt3->close();
            $conn->close();
            
            session_unset();
            session_destroy();
            header('Location: login.php');
            exit;
        }
        
        $user = $user_result->fetch_assoc();
        if ($user['status'] !== 'active') {
            // User account is no longer active
            $stmt3->close();
            $conn->close();
            
            session_unset();
            session_destroy();
            header('Location: login.php');
            exit;
        }
        
        $stmt3->close();
        $_SESSION['last_user_check'] = time();
    }
    
    $conn->close();
    
} catch (Exception $e) {
    // Database error - log it but don't interrupt user
    error_log("Auth check error: " . $e->getMessage());
}
?>