<?php
session_start();
require_once 'config.php';

$page_title = 'Compose Email';

if ($_SERVER["REQUEST_METHOD"] == "POST") {
    $subject = trim($_POST["subject"]);
    $message = trim($_POST["message"]);
    
    // Basic validation
    if (empty($subject) || empty($message)) {
        $_SESSION["error"] = "Subject and message cannot be empty.";
        header("Location: compose.php");
        exit;
    }
    
    if (strlen($subject) > 255) {
        $_SESSION["error"] = "Subject is too long (max 255 characters).";
        header("Location: compose.php");
        exit;
    }
    
    try {
        $conn = getDBConnection();
        
        // Mark previous campaigns as completed
        $conn->query("UPDATE email_campaigns SET status = 'completed' WHERE status != 'completed'");
        
        // Insert new campaign
        $stmt = $conn->prepare("INSERT INTO email_campaigns (subject, message, status) VALUES (?, ?, 'draft')");
        $stmt->bind_param("ss", $subject, $message);
        
        if ($stmt->execute()) {
            $_SESSION["success"] = "Email campaign saved successfully!";
            $_SESSION["campaign_id"] = $conn->insert_id;
        } else {
            throw new Exception("Failed to save campaign.");
        }
        
        $stmt->close();
        $conn->close();
        
    } catch (Exception $e) {
        $_SESSION["error"] = "Error: " . $e->getMessage();
    }
    
    header("Location: compose.php");
    exit;
}

include 'includes/header.php';
?>

<style>
    .editor-toolbar {
        background: var(--light-bg);
        padding: 0.75rem;
        border-radius: 8px 8px 0 0;
        border: 1px solid var(--border-color);
        border-bottom: none;
        display: flex;
        gap: 0.5rem;
        flex-wrap: wrap;
    }
    
    .editor-btn {
        padding: 0.5rem 0.75rem;
        border: 1px solid var(--border-color);
        background: white;
        border-radius: 6px;
        cursor: pointer;
        transition: all 0.3s;
        font-size: 0.875rem;
    }
    
    .editor-btn:hover {
        background: var(--primary-color);
        color: white;
        border-color: var(--primary-color);
    }
    
    textarea.form-control-custom {
        border-radius: 0 0 8px 8px;
        min-height: 400px;
        font-family: 'Courier New', monospace;
    }
    
    .preview-card {
        background: white;
        border-radius: 12px;
        padding: 1.5rem;
        box-shadow: var(--shadow-md);
        border: 1px solid var(--border-color);
    }
    
    .template-btn {
        padding: 1rem;
        border: 2px solid var(--border-color);
        border-radius: 8px;
        background: white;
        transition: all 0.3s;
        cursor: pointer;
        text-align: left;
    }
    
    .template-btn:hover {
        border-color: var(--primary-color);
        background: rgba(79, 70, 229, 0.05);
    }
    
    .template-btn.active {
        border-color: var(--primary-color);
        background: rgba(79, 70, 229, 0.1);
    }
</style>

<div class="page-header">
    <h1 class="page-title">
        <i class="fas fa-edit"></i>
        Compose Email Campaign
    </h1>
    <p class="page-subtitle">Create a compelling email for your recipients</p>
</div>

<?php if (isset($_SESSION["success"])): ?>
<div class="alert alert-custom alert-success-custom alert-dismissible fade show" role="alert">
    <i class="fas fa-check-circle fa-lg"></i>
    <div>
        <strong>Success!</strong> <?php echo htmlspecialchars($_SESSION["success"]); ?>
        <div class="mt-2">
            <a href="sender.php" class="btn btn-sm btn-success-custom btn-custom">
                <i class="fas fa-paper-plane"></i> Start Sending Now
            </a>
        </div>
    </div>
    <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
</div>
<?php unset($_SESSION["success"]); endif; ?>

<?php if (isset($_SESSION["error"])): ?>
<div class="alert alert-custom alert-danger-custom alert-dismissible fade show" role="alert">
    <i class="fas fa-exclamation-circle fa-lg"></i>
    <div><?php echo htmlspecialchars($_SESSION["error"]); ?></div>
    <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
</div>
<?php unset($_SESSION["error"]); endif; ?>

<div class="row">
    <div class="col-lg-8">
        <div class="card-custom">
            <div class="card-header-custom">
                <i class="fas fa-envelope-open-text me-2"></i>
                Email Content
            </div>
            <div class="card-body-custom">
                <form method="POST" action="" id="composeForm">
                    <div class="mb-4">
                        <label class="form-label-custom">
                            <i class="fas fa-tag me-2"></i>
                            Subject Line
                        </label>
                        <input 
                            type="text" 
                            name="subject" 
                            class="form-control form-control-custom" 
                            maxlength="255" 
                            required 
                            placeholder="Enter email subject"
                            id="subjectInput"
                        >
                        <small class="text-muted">
                            <span id="charCount">0</span>/255 characters
                        </small>
                    </div>
                    
                    <div class="mb-4">
                        <label class="form-label-custom">
                            <i class="fas fa-code me-2"></i>
                            Message (HTML supported)
                        </label>
                        
                        <div class="editor-toolbar">
                            <button type="button" class="editor-btn" onclick="insertTag('<strong>', '</strong>')">
                                <i class="fas fa-bold"></i> Bold
                            </button>
                            <button type="button" class="editor-btn" onclick="insertTag('<em>', '</em>')">
                                <i class="fas fa-italic"></i> Italic
                            </button>
                            <button type="button" class="editor-btn" onclick="insertTag('<u>', '</u>')">
                                <i class="fas fa-underline"></i> Underline
                            </button>
                            <button type="button" class="editor-btn" onclick="insertTag('<h1>', '</h1>')">
                                <i class="fas fa-heading"></i> H1
                            </button>
                            <button type="button" class="editor-btn" onclick="insertTag('<h2>', '</h2>')">
                                <i class="fas fa-heading"></i> H2
                            </button>
                            <button type="button" class="editor-btn" onclick="insertTag('<p>', '</p>')">
                                <i class="fas fa-paragraph"></i> Paragraph
                            </button>
                            <button type="button" class="editor-btn" onclick="insertTag('<a href=&quot;&quot;>', '</a>')">
                                <i class="fas fa-link"></i> Link
                            </button>
                            <button type="button" class="editor-btn" onclick="insertTag('<br>', '')">
                                <i class="fas fa-level-down-alt"></i> Break
                            </button>
                        </div>
                        
                        <textarea 
                            name="message" 
                            class="form-control form-control-custom" 
                            required 
                            placeholder="Enter your email message (HTML tags allowed)"
                            id="messageInput"
                        ></textarea>
                    </div>
                    
                    <div class="d-flex gap-2">
                        <button type="submit" class="btn btn-primary-custom btn-custom">
                            <i class="fas fa-save"></i>
                            Save Campaign
                        </button>
                        <button type="button" class="btn btn-outline-custom btn-custom" onclick="previewEmail()">
                            <i class="fas fa-eye"></i>
                            Preview
                        </button>
                    </div>
                </form>
            </div>
        </div>
    </div>
    
    <div class="col-lg-4">
        <div class="card-custom mb-3">
            <div class="card-header-custom">
                <i class="fas fa-lightbulb me-2"></i>
                Quick Tips
            </div>
            <div class="card-body-custom">
                <ul class="list-unstyled mb-0">
                    <li class="mb-2">
                        <i class="fas fa-check text-success me-2"></i>
                        Keep subject lines under 50 characters
                    </li>
                    <li class="mb-2">
                        <i class="fas fa-check text-success me-2"></i>
                        Personalize with recipient's name
                    </li>
                    <li class="mb-2">
                        <i class="fas fa-check text-success me-2"></i>
                        Include a clear call-to-action
                    </li>
                    <li class="mb-2">
                        <i class="fas fa-check text-success me-2"></i>
                        Test on mobile devices
                    </li>
                    <li class="mb-0">
                        <i class="fas fa-check text-success me-2"></i>
                        Avoid spam trigger words
                    </li>
                </ul>
            </div>
        </div>
        
        <div class="card-custom">
            <div class="card-header-custom">
                <i class="fas fa-file-alt me-2"></i>
                Email Templates
            </div>
            <div class="card-body-custom">
                <div class="d-grid gap-2">
                    <button class="template-btn" onclick="loadTemplate('welcome')">
                        <strong>Welcome Email</strong>
                        <small class="d-block text-muted">Greet new subscribers</small>
                    </button>
                    <button class="template-btn" onclick="loadTemplate('newsletter')">
                        <strong>Newsletter</strong>
                        <small class="d-block text-muted">Share updates and news</small>
                    </button>
                    <button class="template-btn" onclick="loadTemplate('promo')">
                        <strong>Promotional</strong>
                        <small class="d-block text-muted">Special offers and deals</small>
                    </button>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Preview Modal -->
<div class="modal fade" id="previewModal" tabindex="-1">
    <div class="modal-dialog modal-lg">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">
                    <i class="fas fa-eye me-2"></i>
                    Email Preview
                </h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body">
                <div class="preview-card">
                    <div class="mb-3">
                        <strong>Subject:</strong>
                        <div id="previewSubject" class="mt-2"></div>
                    </div>
                    <hr>
                    <div>
                        <strong>Message:</strong>
                        <div id="previewMessage" class="mt-2"></div>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<script>
// Character counter
document.getElementById('subjectInput').addEventListener('input', function() {
    document.getElementById('charCount').textContent = this.value.length;
});

// Insert HTML tags
function insertTag(openTag, closeTag) {
    const textarea = document.getElementById('messageInput');
    const start = textarea.selectionStart;
    const end = textarea.selectionEnd;
    const text = textarea.value;
    const selectedText = text.substring(start, end);
    
    textarea.value = text.substring(0, start) + openTag + selectedText + closeTag + text.substring(end);
    textarea.focus();
    textarea.selectionStart = textarea.selectionEnd = start + openTag.length + selectedText.length;
}

// Preview email
function previewEmail() {
    const subject = document.getElementById('subjectInput').value;
    const message = document.getElementById('messageInput').value;
    
    document.getElementById('previewSubject').textContent = subject || '(No subject)';
    document.getElementById('previewMessage').innerHTML = message || '(No message)';
    
    new bootstrap.Modal(document.getElementById('previewModal')).show();
}

// Load templates
function loadTemplate(type) {
    const templates = {
        welcome: {
            subject: 'Welcome to Our Community!',
            message: '<h2>Welcome!</h2>\n<p>Thank you for joining us. We\'re excited to have you on board.</p>\n<p>Here\'s what you can expect:</p>\n<ul>\n<li>Regular updates and news</li>\n<li>Exclusive offers</li>\n<li>Community support</li>\n</ul>\n<p>Best regards,<br>The Team</p>'
        },
        newsletter: {
            subject: 'Monthly Newsletter - Latest Updates',
            message: '<h2>This Month\'s Highlights</h2>\n<p>Dear Subscriber,</p>\n<p>Here are the latest updates from our team:</p>\n<h3>Featured Article</h3>\n<p>[Your content here]</p>\n<h3>Upcoming Events</h3>\n<p>[Event details here]</p>\n<p>Stay tuned for more updates!</p>'
        },
        promo: {
            subject: 'Special Offer - Limited Time Only!',
            message: '<h2>🎉 Exclusive Offer Inside!</h2>\n<p>Don\'t miss out on this special promotion.</p>\n<p><strong>Limited time offer:</strong> Get 25% off on all products!</p>\n<p><a href="#">Claim Your Discount Now</a></p>\n<p>Hurry, offer ends soon!</p>'
        }
    };
    
    if (templates[type]) {
        document.getElementById('subjectInput').value = templates[type].subject;
        document.getElementById('messageInput').value = templates[type].message;
        document.getElementById('charCount').textContent = templates[type].subject.length;
    }
}
</script>

<?php include 'includes/footer.php'; ?>
