<?php
$page_title = 'Cron Sender';
$start_time = microtime(true);
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Cron Sender</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.1/css/all.min.css">
    <style>
        body {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            padding: 20px;
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
        }
        
        .container {
            max-width: 900px;
            margin: 0 auto;
        }
        
        .card {
            background: white;
            border-radius: 20px;
            box-shadow: 0 20px 60px rgba(0, 0, 0, 0.3);
            overflow: hidden;
            animation: slideUp 0.5s ease-out;
        }
        
        @keyframes slideUp {
            from { opacity: 0; transform: translateY(30px); }
            to { opacity: 1; transform: translateY(0); }
        }
        
        .card-header {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            padding: 30px;
            text-align: center;
        }
        
        .card-header h1 {
            margin: 0;
            font-size: 2rem;
            font-weight: 700;
        }
        
        .card-body {
            padding: 0;
        }
        
        .terminal {
            background: #1e1e1e;
            color: #d4d4d4;
            padding: 20px;
            font-family: 'Courier New', monospace;
            font-size: 14px;
            line-height: 1.6;
            max-height: 500px;
            overflow-y: auto;
        }
        
        .terminal::-webkit-scrollbar {
            width: 8px;
        }
        
        .terminal::-webkit-scrollbar-track {
            background: #2d2d2d;
        }
        
        .terminal::-webkit-scrollbar-thumb {
            background: #667eea;
            border-radius: 4px;
        }
        
        .log-line {
            margin: 4px 0;
            padding: 4px 8px;
            border-radius: 4px;
            transition: background 0.3s;
        }
        
        .log-line:hover {
            background: rgba(255, 255, 255, 0.05);
        }
        
        .timestamp {
            color: #858585;
            font-weight: bold;
        }
        
        .success {
            color: #4ec9b0;
        }
        
        .error {
            color: #f48771;
            background: rgba(244, 135, 113, 0.1);
        }
        
        .info {
            color: #dcdcaa;
        }
        
        .warning {
            color: #ce9178;
        }
        
        .highlight {
            color: #c586c0;
            font-weight: bold;
        }
        
        .stats-row {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(150px, 1fr));
            gap: 15px;
            padding: 20px;
            background: #f8f9fa;
        }
        
        .stat-box {
            text-align: center;
            padding: 20px;
            background: white;
            border-radius: 12px;
            box-shadow: 0 2px 8px rgba(0, 0, 0, 0.1);
        }
        
        .stat-number {
            font-size: 2.5rem;
            font-weight: 700;
            margin: 10px 0;
        }
        
        .stat-label {
            color: #6c757d;
            font-size: 0.9rem;
            text-transform: uppercase;
            letter-spacing: 1px;
        }
        
        .stat-success .stat-number {
            color: #28a745;
        }
        
        .stat-error .stat-number {
            color: #dc3545;
        }
        
        .stat-info .stat-number {
            color: #17a2b8;
        }
        
        .stat-time .stat-number {
            color: #6f42c1;
            font-size: 1.5rem;
        }
        
        .actions {
            padding: 20px;
            text-align: center;
            background: white;
            border-top: 1px solid #e9ecef;
        }
        
        .btn-custom {
            padding: 12px 30px;
            border-radius: 10px;
            font-weight: 600;
            text-decoration: none;
            display: inline-block;
            margin: 5px;
            transition: all 0.3s;
        }
        
        .btn-primary-custom {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            border: none;
        }
        
        .btn-primary-custom:hover {
            transform: translateY(-2px);
            box-shadow: 0 10px 25px rgba(102, 126, 234, 0.4);
            color: white;
        }
        
        .btn-success-custom {
            background: linear-gradient(135deg, #28a745 0%, #20c997 100%);
            color: white;
            border: none;
        }
        
        .btn-success-custom:hover {
            transform: translateY(-2px);
            box-shadow: 0 10px 25px rgba(40, 167, 69, 0.4);
            color: white;
        }
        
        .loading {
            display: inline-block;
            width: 20px;
            height: 20px;
            border: 3px solid rgba(255,255,255,.3);
            border-radius: 50%;
            border-top-color: #fff;
            animation: spin 1s ease-in-out infinite;
        }
        
        @keyframes spin {
            to { transform: rotate(360deg); }
        }
    </style>
</head>
<body>
    <div class="container">
        <div class="card">
            <div class="card-header">
                <i class="fas fa-terminal fa-2x mb-3"></i>
                <h1>🧪 Cron Sender Test</h1>
                <p class="mb-0">Real-time Email Sending Monitor</p>
            </div>
            
            <div class="stats-row" id="statsRow">
                <div class="stat-box stat-success">
                    <i class="fas fa-check-circle fa-2x mb-2 text-success"></i>
                    <div class="stat-number" id="sentCount">0</div>
                    <div class="stat-label">Sent</div>
                </div>
                <div class="stat-box stat-error">
                    <i class="fas fa-times-circle fa-2x mb-2 text-danger"></i>
                    <div class="stat-number" id="failedCount">0</div>
                    <div class="stat-label">Failed</div>
                </div>
                <div class="stat-box stat-info">
                    <i class="fas fa-envelope fa-2x mb-2 text-info"></i>
                    <div class="stat-number" id="totalCount">0</div>
                    <div class="stat-label">Total</div>
                </div>
                <div class="stat-box stat-time">
                    <i class="fas fa-clock fa-2x mb-2 text-purple"></i>
                    <div class="stat-number" id="timeCount">0.0s</div>
                    <div class="stat-label">Elapsed</div>
                </div>
            </div>
            
            <div class="card-body">
                <div class="terminal" id="terminal">
<?php
// Start output buffering
ob_start();

// Basic error handling
error_reporting(E_ALL);
ini_set('display_errors', 1);
ini_set('memory_limit', '256M');
set_time_limit(300);

$sent_count = 0;
$failed_count = 0;

// Enhanced logging function
function logMessage($message, $type = 'info') {
    global $sent_count, $failed_count;
    
    $timestamp = date('Y-m-d H:i:s');
    $icon = '•';
    $class = $type;
    
    if (strpos($message, '✓') !== false || strpos($message, 'Complete') !== false || strpos($message, 'Success') !== false) {
        $icon = '✓';
        $class = 'success';
    } elseif (strpos($message, '✗') !== false || strpos($message, 'ERROR') !== false || strpos($message, 'Failed') !== false) {
        $icon = '✗';
        $class = 'error';
    } elseif (strpos($message, 'Campaign:') !== false || strpos($message, 'Processing') !== false) {
        $icon = '★';
        $class = 'highlight';
    }
    
    // Extract counts for live updating
    if (preg_match('/Sent: (\d+)/', $message, $matches)) {
        $sent_count = $matches[1];
    }
    if (preg_match('/Failed: (\d+)/', $message, $matches)) {
        $failed_count = $matches[1];
    }
    
    echo "<div class='log-line $class'>";
    echo "<span class='timestamp'>[$timestamp]</span> ";
    echo "<span style='color: " . ($class == 'success' ? '#4ec9b0' : ($class == 'error' ? '#f48771' : '#dcdcaa')) . ";'>$icon</span> ";
    echo htmlspecialchars($message);
    echo "</div>\n";
    
    // Update stats via JavaScript
    if ($class == 'success' || $class == 'error') {
        echo "<script>
            document.getElementById('sentCount').textContent = '$sent_count';
            document.getElementById('failedCount').textContent = '$failed_count';
            document.getElementById('totalCount').textContent = '" . ($sent_count + $failed_count) . "';
        </script>";
    }
    
    flush();
    ob_flush();
}

logMessage("=== Cron Email Sender Test Started ===", 'info');

// Load config
if (!file_exists(__DIR__ . '/config.php')) {
    logMessage("ERROR: config.php not found", 'error');
    die();
}

require_once __DIR__ . '/config.php';
logMessage("Config loaded successfully", 'success');

// Load PHPMailer
if (!file_exists(__DIR__ . '/PHPMailer/PHPMailer.php')) {
    logMessage("ERROR: PHPMailer not found", 'error');
    die();
}

require __DIR__ . '/PHPMailer/PHPMailer.php';
require __DIR__ . '/PHPMailer/SMTP.php';
require __DIR__ . '/PHPMailer/Exception.php';

use PHPMailer\PHPMailer\PHPMailer;
use PHPMailer\PHPMailer\Exception;

logMessage("PHPMailer loaded successfully", 'success');

try {
    // Connect to database
    $conn = getDBConnection();
    logMessage("Database connected successfully", 'success');
    
    // Get active campaign
    $campaign_result = $conn->query("SELECT * FROM email_campaigns WHERE status != 'completed' ORDER BY id DESC LIMIT 1");
    
    if (!$campaign_result || $campaign_result->num_rows === 0) {
        logMessage("No active campaign found", 'warning');
        echo "<div class='log-line info'><i class='fas fa-info-circle'></i> To fix: Create a campaign in compose.php</div>\n";
        $conn->close();
        exit(0);
    }
    
    $campaign = $campaign_result->fetch_assoc();
    logMessage("Campaign: ID #{$campaign['id']} - {$campaign['subject']}", 'info');
    
    // Get pending emails
    $batch_size = defined('BATCH_SIZE') ? BATCH_SIZE : 100;
    $pending = $conn->query("SELECT * FROM email_lists WHERE status = 'pending' LIMIT $batch_size");
    
    if ($pending->num_rows == 0) {
        logMessage("No pending emails found", 'warning');
        echo "<div class='log-line info'><i class='fas fa-info-circle'></i> To fix: Upload emails via upload.php or reset via reset_to_pending.php</div>\n";
        
        $stmt = $conn->prepare("UPDATE email_campaigns SET status = 'completed' WHERE id = ?");
        $stmt->bind_param("i", $campaign["id"]);
        $stmt->execute();
        $stmt->close();
        
        $conn->close();
        logMessage("Campaign marked as completed", 'success');
        exit(0);
    }
    
    logMessage("Processing {$pending->num_rows} emails...", 'info');
    echo "<div class='log-line info'>━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━</div>\n";
    
    $sent = 0;
    $failed = 0;
    
    // Send emails
    while ($row = $pending->fetch_assoc()) {
        try {
            $mail = new PHPMailer(true);
            
            // SMTP settings
            $mail->isSMTP();
            $mail->Host = SMTP_HOST;
            $mail->SMTPAuth = true;
            $mail->Username = SMTP_USERNAME;
            $mail->Password = SMTP_PASSWORD;
            $mail->SMTPSecure = SMTP_ENCRYPTION;
            $mail->Port = SMTP_PORT;
            $mail->SMTPDebug = 0;
            
            // Email content
            $mail->setFrom(FROM_EMAIL, FROM_NAME);
            $mail->addAddress($row["email"], $row["name"]);
            $mail->isHTML(true);
            $mail->Subject = $campaign["subject"];
            $mail->Body = $campaign["message"];
            
            // Send
            $mail->send();
            
            // Update database
            $stmt = $conn->prepare("UPDATE email_lists SET status = 'sent', sent_date = NOW() WHERE id = ?");
            $stmt->bind_param("i", $row["id"]);
            $stmt->execute();
            $stmt->close();
            
            $sent++;
            logMessage("✓ Sent successfully to: {$row['email']}", 'success');
            
            // Small delay
            usleep(500000); // 0.5 seconds
            
        } catch (Exception $e) {
            $error = substr($e->getMessage(), 0, 200);
            
            $stmt = $conn->prepare("UPDATE email_lists SET status = 'failed', error_message = ? WHERE id = ?");
            $stmt->bind_param("si", $error, $row["id"]);
            $stmt->execute();
            $stmt->close();
            
            $failed++;
            logMessage("✗ Failed to send to {$row['email']}: $error", 'error');
        }
    }
    
    $conn->close();
    
    // Summary
    echo "<div class='log-line info'>━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━</div>\n";
    logMessage("=== Batch Complete ===", 'success');
    logMessage("Sent: $sent", 'success');
    logMessage("Failed: $failed", $failed > 0 ? 'error' : 'info');
    logMessage("Total Processed: " . ($sent + $failed), 'info');
    
    $end_time = microtime(true);
    $duration = round($end_time - $start_time, 2);
    logMessage("Execution Time: {$duration}s", 'info');
    
    echo "<script>document.getElementById('timeCount').textContent = '{$duration}s';</script>";
    
} catch (Exception $e) {
    logMessage("CRITICAL ERROR: " . $e->getMessage(), 'error');
    if (isset($conn)) $conn->close();
}

ob_end_flush();
?>
                </div>
            </div>
            
            <div class="actions">
                <a href="index.php" class="btn-custom btn-primary-custom">
                    <i class="fas fa-home"></i> Dashboard
                </a>
                <a href="test_cron.php" class="btn-custom btn-success-custom">
                    <i class="fas fa-redo"></i> Run Again
                </a>
                <a href="upload.php" class="btn-custom btn-primary-custom">
                    <i class="fas fa-upload"></i> Upload More
                </a>
            </div>
        </div>
    </div>
    
    <script>
        // Auto-scroll terminal to bottom
        const terminal = document.getElementById('terminal');
        terminal.scrollTop = terminal.scrollHeight;
        
        // Animate numbers
        function animateValue(id, start, end, duration) {
            const obj = document.getElementById(id);
            let current = start;
            const range = end - start;
            const increment = end > start ? 1 : -1;
            const stepTime = Math.abs(Math.floor(duration / range));
            const timer = setInterval(function() {
                current += increment;
                obj.textContent = current;
                if (current == end) {
                    clearInterval(timer);
                }
            }, stepTime);
        }
    </script>
</body>
</html>