<?php
/**
 * Single Device Login Diagnostic Tool
 * This will check if the feature is properly installed
 */
session_start();
require_once 'config.php';

$diagnostics = [];
$all_good = true;

// Check 1: Database tables
$conn = getDBConnection();

echo "<h2>🔍 Single Device Login Diagnostic</h2>";
echo "<style>
body { font-family: Arial, sans-serif; padding: 20px; background: #f5f5f5; }
.check { padding: 15px; margin: 10px 0; border-radius: 5px; }
.pass { background: #d4edda; border-left: 4px solid #28a745; }
.fail { background: #f8d7da; border-left: 4px solid #dc3545; }
.info { background: #d1ecf1; border-left: 4px solid #17a2b8; }
pre { background: #f8f9fa; padding: 10px; border-radius: 4px; overflow-x: auto; }
h2 { color: #333; }
</style>";

// Check if user_sessions table exists
$tables_check = $conn->query("SHOW TABLES LIKE 'user_sessions'");
if ($tables_check->num_rows > 0) {
    echo "<div class='check pass'>✓ user_sessions table exists</div>";
    
    // Check table structure
    $columns = $conn->query("DESCRIBE user_sessions");
    echo "<div class='check info'>Table structure:<br>";
    echo "<pre>";
    while ($col = $columns->fetch_assoc()) {
        echo $col['Field'] . " - " . $col['Type'] . "\n";
    }
    echo "</pre></div>";
    
    // Check for active sessions
    $active_sessions = $conn->query("SELECT COUNT(*) as count FROM user_sessions WHERE expires_at > NOW()")->fetch_assoc();
    echo "<div class='check info'>Active sessions in database: " . $active_sessions['count'] . "</div>";
    
    // Show all sessions with details
    $all_sessions = $conn->query("SELECT s.*, u.username FROM user_sessions s JOIN users u ON s.user_id = u.id WHERE expires_at > NOW()");
    if ($all_sessions->num_rows > 0) {
        echo "<div class='check info'><strong>Current Active Sessions:</strong><br>";
        echo "<table border='1' style='width:100%; border-collapse: collapse; margin-top: 10px;'>";
        echo "<tr><th>Username</th><th>Session ID</th><th>IP Address</th><th>Last Activity</th><th>Expires At</th></tr>";
        while ($sess = $all_sessions->fetch_assoc()) {
            echo "<tr>";
            echo "<td>" . htmlspecialchars($sess['username']) . "</td>";
            echo "<td><small>" . substr($sess['session_id'], 0, 20) . "...</small></td>";
            echo "<td>" . htmlspecialchars($sess['ip_address']) . "</td>";
            echo "<td>" . $sess['last_activity'] . "</td>";
            echo "<td>" . $sess['expires_at'] . "</td>";
            echo "</tr>";
        }
        echo "</table></div>";
    }
    
} else {
    echo "<div class='check fail'>✗ user_sessions table NOT found! Database not imported correctly.</div>";
    $all_good = false;
}

// Check if login.php has the new code
$login_file = file_get_contents('login.php');
if (strpos($login_file, 'force_login') !== false && strpos($login_file, 'existing_session') !== false) {
    echo "<div class='check pass'>✓ login.php has single-device code</div>";
} else {
    echo "<div class='check fail'>✗ login.php is NOT updated! Still using old code.</div>";
    $all_good = false;
}

// Check if auth_check.php has the new code
if (file_exists('auth_check.php')) {
    $auth_file = file_get_contents('auth_check.php');
    if (strpos($auth_file, 'current_session_id') !== false && strpos($auth_file, 'user_sessions') !== false) {
        echo "<div class='check pass'>✓ auth_check.php has session validation code</div>";
    } else {
        echo "<div class='check fail'>✗ auth_check.php is NOT updated! Still using old code.</div>";
        $all_good = false;
    }
} else {
    echo "<div class='check fail'>✗ auth_check.php file not found!</div>";
    $all_good = false;
}

// Check for multiple sessions per user
$multi_sessions = $conn->query("
    SELECT u.username, COUNT(s.id) as session_count
    FROM users u
    JOIN user_sessions s ON u.id = s.user_id
    WHERE s.expires_at > NOW()
    GROUP BY u.id, u.username
    HAVING session_count > 1
");

if ($multi_sessions->num_rows > 0) {
    echo "<div class='check fail'><strong>⚠️ PROBLEM DETECTED: Multiple sessions found!</strong><br>";
    while ($user = $multi_sessions->fetch_assoc()) {
        echo "User '" . htmlspecialchars($user['username']) . "' has " . $user['session_count'] . " active sessions<br>";
    }
    echo "</div>";
    $all_good = false;
} else {
    echo "<div class='check pass'>✓ No multiple sessions detected (all users have max 1 session)</div>";
}

// Test query
echo "<div class='check info'><strong>Test Query:</strong> Checking if session detection works<br>";
if (isset($_SESSION['user_id'])) {
    $test_query = "SELECT session_id FROM user_sessions WHERE user_id = " . $_SESSION['user_id'] . " AND expires_at > NOW()";
    echo "<pre>$test_query</pre>";
    $result = $conn->query($test_query);
    echo "Result: " . $result->num_rows . " session(s) found for current user";
    echo "</div>";
} else {
    echo "Not logged in - cannot test current user</div>";
}

// Summary
echo "<hr>";
if ($all_good) {
    echo "<div class='check pass'><h3>✅ All checks passed! Feature should be working.</h3>";
    echo "<p><strong>If you can still login on 2 devices:</strong></p>";
    echo "<ol>";
    echo "<li>Clear browser cache on both devices</li>";
    echo "<li>Logout from both devices</li>";
    echo "<li>Login again on Device 1</li>";
    echo "<li>Try login on Device 2 - you should see warning</li>";
    echo "</ol></div>";
} else {
    echo "<div class='check fail'><h3>❌ Issues detected! Feature is NOT working properly.</h3>";
    echo "<p><strong>To fix:</strong></p>";
    echo "<ol>";
    echo "<li>Make sure database_users.sql was imported to phpMyAdmin</li>";
    echo "<li>Replace login.php with login_single_device.php</li>";
    echo "<li>Replace auth_check.php with auth_check_single_device.php</li>";
    echo "<li>Clear browser cache</li>";
    echo "<li>Test again</li>";
    echo "</ol></div>";
}

// Clean up tool
echo "<hr>";
echo "<h3>🧹 Cleanup Tool</h3>";
echo "<p>If you want to remove duplicate sessions and enforce single device:</p>";
echo "<form method='post' action='' onsubmit='return confirm(\"This will keep only the MOST RECENT session for each user. Continue?\");'>";
echo "<input type='hidden' name='cleanup' value='1'>";
echo "<button type='submit' style='padding: 10px 20px; background: #dc3545; color: white; border: none; border-radius: 5px; cursor: pointer;'>Delete All Duplicate Sessions</button>";
echo "</form>";

if (isset($_POST['cleanup'])) {
    // Get all users with multiple sessions
    $users_multi = $conn->query("
        SELECT user_id, COUNT(*) as count 
        FROM user_sessions 
        WHERE expires_at > NOW() 
        GROUP BY user_id 
        HAVING count > 1
    ");
    
    $cleaned = 0;
    while ($user = $users_multi->fetch_assoc()) {
        // Keep only the most recent session
        $conn->query("
            DELETE FROM user_sessions 
            WHERE user_id = {$user['user_id']} 
            AND id NOT IN (
                SELECT id FROM (
                    SELECT id FROM user_sessions 
                    WHERE user_id = {$user['user_id']} 
                    ORDER BY last_activity DESC 
                    LIMIT 1
                ) as keep
            )
        ");
        $cleaned++;
    }
    
    echo "<div class='check pass'>✓ Cleaned up duplicate sessions for $cleaned users. Refresh page to see results.</div>";
}

$conn->close();
?>

<hr>
<p><a href="index.php">← Back to Dashboard</a> | <a href="active_sessions.php">View Active Sessions</a> | <a href="?">Refresh Diagnostic</a></p>