<?php
session_start();
require_once 'config.php';

// Check if already logged in
if (isset($_SESSION['user_id'])) {
    header('Location: index.php');
    exit;
}

$error = '';
$success = '';
$show_force_login = false;

// Check for messages
if (isset($_GET['logged_out'])) {
    $success = 'You have been successfully logged out.';
}

if (isset($_GET['timeout'])) {
    $error = 'Your session has expired. Please login again.';
}

if (isset($_GET['kicked'])) {
    $error = 'You have been logged out because someone logged in on another device.';
}

if (isset($_GET['registered'])) {
    $success = 'Account created successfully! Please login.';
}

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $username = trim($_POST['username'] ?? '');
    $password = $_POST['password'] ?? '';
    $force_login = isset($_POST['force_login']) && $_POST['force_login'] === '1';
    
    if (empty($username) || empty($password)) {
        $error = 'Please enter both username and password';
    } else {
        $conn = getDBConnection();
        
        // Get user from database
        $stmt = $conn->prepare("SELECT id, username, email, password, full_name, status, role, login_attempts, locked_until, license_key FROM users WHERE username = ? OR email = ?");
        $stmt->bind_param("ss", $username, $username);
        $stmt->execute();
        $result = $stmt->get_result();
        
        if ($result->num_rows === 0) {
            $error = 'Invalid username or password';
            sleep(2);
        } else {
            $user = $result->fetch_assoc();
            
            // Check if account is locked
            if ($user['locked_until'] && strtotime($user['locked_until']) > time()) {
                $minutes_left = ceil((strtotime($user['locked_until']) - time()) / 60);
                $error = "Account is locked. Please try again in $minutes_left minute(s).";
            }
            // Check if account is active
            elseif ($user['status'] !== 'active') {
                $error = 'Your account is not active. Please contact administrator.';
            }
            // Verify password
            elseif (password_verify($password, $user['password'])) {
                
                // CHECK FOR EXISTING ACTIVE SESSION
                $stmt2 = $conn->prepare("SELECT session_id, ip_address, user_agent, last_activity FROM user_sessions WHERE user_id = ? AND expires_at > NOW() ORDER BY last_activity DESC LIMIT 1");
                $stmt2->bind_param("i", $user['id']);
                $stmt2->execute();
                $existing_session = $stmt2->get_result()->fetch_assoc();
                $stmt2->close();
                
                // If there's an active session and not forcing login, show warning
                if ($existing_session && !$force_login) {
                    $last_active = date('M j, Y g:i A', strtotime($existing_session['last_activity']));
                    $error = "Your account is already logged in on another device!<br>Last active: $last_active<br>IP: {$existing_session['ip_address']}";
                    $show_force_login = true;
                } else {
                    // CRITICAL FIX: ALWAYS DELETE ALL OLD SESSIONS BEFORE CREATING NEW ONE
                    // This ensures only ONE device can be logged in at a time
                    $stmt3 = $conn->prepare("DELETE FROM user_sessions WHERE user_id = ?");
                    $stmt3->bind_param("i", $user['id']);
                    $stmt3->execute();
                    $stmt3->close();
                    
                    // Login successful - Create new session
                    $_SESSION['user_id'] = $user['id'];
                    $_SESSION['username'] = $user['username'];
                    $_SESSION['email'] = $user['email'];
                    $_SESSION['full_name'] = $user['full_name'];
                    $_SESSION['role'] = $user['role'];
                    $_SESSION['license_key'] = $user['license_key'];
                    $_SESSION['login_time'] = time();
                    $_SESSION['last_activity'] = time();
                    
                    // Reset login attempts
                    $stmt2 = $conn->prepare("UPDATE users SET login_attempts = 0, locked_until = NULL, last_login = NOW() WHERE id = ?");
                    $stmt2->bind_param("i", $user['id']);
                    $stmt2->execute();
                    $stmt2->close();
                    
                    // Create session record in database
                    $session_id = session_id();
                    $ip_address = $_SERVER['REMOTE_ADDR'] ?? 'unknown';
                    $user_agent = substr($_SERVER['HTTP_USER_AGENT'] ?? 'unknown', 0, 255);
                    $expires_at = date('Y-m-d H:i:s', time() + 1800); // 30 minutes
                    
                    $stmt3 = $conn->prepare("INSERT INTO user_sessions (user_id, session_id, ip_address, user_agent, last_activity, expires_at) VALUES (?, ?, ?, ?, NOW(), ?)");
                    $stmt3->bind_param("issss", $user['id'], $session_id, $ip_address, $user_agent, $expires_at);
                    $stmt3->execute();
                    $stmt3->close();
                    
                    $conn->close();
                    
                    // Redirect to dashboard
                    header('Location: index.php');
                    exit;
                }
            } else {
                // Wrong password - increment login attempts
                $attempts = $user['login_attempts'] + 1;
                
                if ($attempts >= 5) {
                    // Lock account for 30 minutes
                    $locked_until = date('Y-m-d H:i:s', time() + 1800);
                    $stmt2 = $conn->prepare("UPDATE users SET login_attempts = ?, locked_until = ? WHERE id = ?");
                    $stmt2->bind_param("isi", $attempts, $locked_until, $user['id']);
                    $stmt2->execute();
                    $stmt2->close();
                    $error = 'Too many failed attempts. Account locked for 30 minutes.';
                } else {
                    $stmt2 = $conn->prepare("UPDATE users SET login_attempts = ? WHERE id = ?");
                    $stmt2->bind_param("ii", $attempts, $user['id']);
                    $stmt2->execute();
                    $stmt2->close();
                    $remaining = 5 - $attempts;
                    $error = "Invalid username or password. $remaining attempt(s) remaining.";
                }
                
                sleep(2);
            }
        }
        
        $stmt->close();
        $conn->close();
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Login - Email Sender</title>
    
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.1/css/all.min.css">
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@400;500;600;700&display=swap" rel="stylesheet">
    
    <style>
        * {
            font-family: 'Inter', sans-serif;
        }
        
        body {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            display: flex;
            align-items: center;
            justify-content: center;
            padding: 20px;
        }
        
        .login-container {
            width: 100%;
            max-width: 450px;
        }
        
        .login-card {
            background: white;
            border-radius: 20px;
            box-shadow: 0 20px 60px rgba(0, 0, 0, 0.3);
            overflow: hidden;
            animation: slideUp 0.5s ease-out;
        }
        
        @keyframes slideUp {
            from {
                opacity: 0;
                transform: translateY(30px);
            }
            to {
                opacity: 1;
                transform: translateY(0);
            }
        }
        
        .login-header {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            padding: 40px 30px;
            text-align: center;
        }
        
        .login-icon {
            width: 80px;
            height: 80px;
            background: rgba(255, 255, 255, 0.2);
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            margin: 0 auto 20px;
            animation: pulse 2s infinite;
        }
        
        @keyframes pulse {
            0%, 100% {
                transform: scale(1);
            }
            50% {
                transform: scale(1.05);
            }
        }
        
        .login-icon i {
            font-size: 36px;
        }
        
        .login-header h1 {
            margin: 0;
            font-size: 1.8rem;
            font-weight: 700;
        }
        
        .login-header p {
            margin: 10px 0 0 0;
            opacity: 0.9;
        }
        
        .login-body {
            padding: 40px 30px;
        }
        
        .form-group {
            margin-bottom: 25px;
        }
        
        .form-label {
            font-weight: 600;
            color: #333;
            margin-bottom: 8px;
            display: block;
        }
        
        .form-control {
            border: 2px solid #e5e7eb;
            border-radius: 10px;
            padding: 12px 15px;
            font-size: 15px;
            transition: all 0.3s;
        }
        
        .form-control:focus {
            border-color: #667eea;
            box-shadow: 0 0 0 4px rgba(102, 126, 234, 0.1);
            outline: none;
        }
        
        .input-group {
            position: relative;
        }
        
        .input-group i {
            position: absolute;
            left: 15px;
            top: 50%;
            transform: translateY(-50%);
            color: #9ca3af;
            z-index: 10;
        }
        
        .input-group .form-control {
            padding-left: 45px;
        }
        
        .btn-login {
            width: 100%;
            padding: 14px;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            border: none;
            border-radius: 10px;
            font-size: 16px;
            font-weight: 600;
            transition: all 0.3s;
            cursor: pointer;
        }
        
        .btn-login:hover {
            transform: translateY(-2px);
            box-shadow: 0 10px 25px rgba(102, 126, 234, 0.4);
        }
        
        .btn-force {
            width: 100%;
            padding: 14px;
            background: linear-gradient(135deg, #ef4444 0%, #dc2626 100%);
            color: white;
            border: none;
            border-radius: 10px;
            font-size: 16px;
            font-weight: 600;
            transition: all 0.3s;
            cursor: pointer;
            margin-top: 10px;
        }
        
        .btn-force:hover {
            transform: translateY(-2px);
            box-shadow: 0 10px 25px rgba(239, 68, 68, 0.4);
        }
        
        .alert {
            border-radius: 10px;
            padding: 12px 15px;
            margin-bottom: 20px;
            border: none;
        }
        
        .alert-danger {
            background: #fee2e2;
            color: #991b1b;
        }
        
        .alert-success {
            background: #d1fae5;
            color: #065f46;
        }
        
        .alert-warning {
            background: #fef3c7;
            color: #92400e;
            border: 2px solid #fbbf24;
        }
        
        .footer-text {
            text-align: center;
            color: #6b7280;
            font-size: 14px;
            margin-top: 20px;
        }
        
        .footer-text a {
            color: #667eea;
            text-decoration: none;
            font-weight: 600;
        }
        
        .footer-text a:hover {
            text-decoration: underline;
        }
        
        .toggle-password {
            position: absolute;
            right: 15px;
            top: 50%;
            transform: translateY(-50%);
            cursor: pointer;
            color: #9ca3af;
            z-index: 10;
        }
        
        .toggle-password:hover {
            color: #667eea;
        }
        
        .device-info {
            background: #f3f4f6;
            padding: 15px;
            border-radius: 10px;
            margin-bottom: 15px;
            font-size: 14px;
        }
        
        @media (max-width: 576px) {
            .login-header {
                padding: 30px 20px;
            }
            
            .login-body {
                padding: 30px 20px;
            }
            
            .login-header h1 {
                font-size: 1.5rem;
            }
        }
    </style>
</head>
<body>
    <div class="login-container">
        <div class="login-card">
            <div class="login-header">
                <div class="login-icon">
                    <i class="fas fa-shield-alt"></i>
                </div>
                <h1>Welcome Back</h1>
                <p>Email Sender System</p>
            </div>
            
            <div class="login-body">
                <?php if ($success): ?>
                <div class="alert alert-success">
                    <i class="fas fa-check-circle me-2"></i>
                    <?php echo $success; ?>
                </div>
                <?php endif; ?>
                
                <?php if ($error && !$show_force_login): ?>
                <div class="alert alert-danger">
                    <i class="fas fa-exclamation-circle me-2"></i>
                    <?php echo $error; ?>
                </div>
                <?php endif; ?>
                
                <?php if ($show_force_login): ?>
                <div class="alert alert-warning">
                    <i class="fas fa-exclamation-triangle me-2"></i>
                    <strong>Active Session Detected!</strong><br>
                    <?php echo $error; ?>
                </div>
                
                <div class="device-info">
                    <i class="fas fa-info-circle me-2"></i>
                    <strong>Note:</strong> Forcing login will log out the other device immediately.
                </div>
                
                <form method="POST" action="">
                    <input type="hidden" name="username" value="<?php echo htmlspecialchars($_POST['username'] ?? ''); ?>">
                    <input type="hidden" name="password" value="<?php echo htmlspecialchars($_POST['password'] ?? ''); ?>">
                    <input type="hidden" name="force_login" value="1">
                    
                    <button type="submit" class="btn-force">
                        <i class="fas fa-sign-in-alt me-2"></i>
                        Force Login & Logout Other Device
                    </button>
                </form>
                
                <div class="text-center mt-3">
                    <a href="login.php" class="btn btn-outline-secondary">
                        <i class="fas fa-arrow-left me-2"></i>
                        Cancel
                    </a>
                </div>
                
                <?php else: ?>
                
                <form method="POST" action="">
                    <div class="form-group">
                        <label class="form-label">
                            <i class="fas fa-user me-2"></i>
                            Username or Email
                        </label>
                        <div class="input-group">
                            <i class="fas fa-user"></i>
                            <input 
                                type="text" 
                                name="username" 
                                class="form-control" 
                                placeholder="Enter username or email" 
                                required 
                                autofocus
                                value="<?php echo htmlspecialchars($_POST['username'] ?? ''); ?>"
                            >
                        </div>
                    </div>
                    
                    <div class="form-group">
                        <label class="form-label">
                            <i class="fas fa-lock me-2"></i>
                            Password
                        </label>
                        <div class="input-group">
                            <i class="fas fa-lock"></i>
                            <input 
                                type="password" 
                                name="password" 
                                id="password"
                                class="form-control" 
                                placeholder="Enter your password" 
                                required
                            >
                            <i class="fas fa-eye toggle-password" id="togglePassword"></i>
                        </div>
                    </div>
                    
                    <button type="submit" class="btn-login">
                        <i class="fas fa-sign-in-alt me-2"></i>
                        Login
                    </button>
                </form>
                
                <div class="footer-text">
                    <i class="fas fa-lock me-1"></i>
                    Secure Access • One Device Per License
                    <br><br>
                    Don't have an account? <a href="signup.php">Sign up here</a>
                </div>
                
                <?php endif; ?>
            </div>
        </div>
    </div>
    
    <script>
        const togglePassword = document.getElementById('togglePassword');
        const password = document.getElementById('password');
        
        if (togglePassword && password) {
            togglePassword.addEventListener('click', function() {
                const type = password.getAttribute('type') === 'password' ? 'text' : 'password';
                password.setAttribute('type', type);
                this.classList.toggle('fa-eye');
                this.classList.toggle('fa-eye-slash');
            });
        }
    </script>
</body>
</html>