<?php
/**
 * Debug Login - This will show you exactly what's happening
 */
session_start();
require_once 'config.php';

// Debug mode - shows all SQL queries
$debug_messages = [];

function add_debug($message) {
    global $debug_messages;
    $debug_messages[] = date('H:i:s') . " - " . $message;
}

add_debug("Script started");

// Check if already logged in
if (isset($_SESSION['user_id'])) {
    add_debug("User already logged in, redirecting");
    header('Location: index.php');
    exit;
}

$error = '';
$success = '';
$show_force_login = false;
$temp_user_id = null;

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    add_debug("POST request received");
    
    $username = trim($_POST['username'] ?? '');
    $password = $_POST['password'] ?? '';
    $force_login = isset($_POST['force_login']) && $_POST['force_login'] === '1';
    
    add_debug("Username: $username, Force: " . ($force_login ? 'YES' : 'NO'));
    
    if (empty($username) || empty($password)) {
        $error = 'Please enter both username and password';
        add_debug("Empty credentials");
    } else {
        $conn = getDBConnection();
        add_debug("Database connected");
        
        // Get user from database
        $stmt = $conn->prepare("SELECT id, username, email, password, full_name, status, role, login_attempts, locked_until, license_key FROM users WHERE username = ? OR email = ?");
        $stmt->bind_param("ss", $username, $username);
        $stmt->execute();
        $result = $stmt->get_result();
        
        add_debug("User query executed, rows: " . $result->num_rows);
        
        if ($result->num_rows === 0) {
            $error = 'Invalid username or password';
            add_debug("User not found");
        } else {
            $user = $result->fetch_assoc();
            add_debug("User found: ID=" . $user['id'] . ", Status=" . $user['status']);
            
            // Check status
            if ($user['status'] !== 'active') {
                $error = 'Your account is not active.';
                add_debug("Account not active");
            }
            // Verify password
            elseif (password_verify($password, $user['password'])) {
                add_debug("Password verified successfully");
                
                // CHECK FOR EXISTING SESSIONS
                $stmt2 = $conn->prepare("SELECT id, session_id, ip_address FROM user_sessions WHERE user_id = ? AND expires_at > NOW()");
                $stmt2->bind_param("i", $user['id']);
                $stmt2->execute();
                $session_result = $stmt2->get_result();
                $existing_session = $session_result->fetch_assoc();
                $stmt2->close();
                
                add_debug("Existing sessions check: " . ($existing_session ? "FOUND (ID: {$existing_session['id']})" : "NONE"));
                
                if ($existing_session && !$force_login) {
                    add_debug("Showing force login prompt");
                    $error = "⚠️ Account already logged in!<br>IP: {$existing_session['ip_address']}";
                    $show_force_login = true;
                    $temp_user_id = $user['id'];
                } else {
                    add_debug("Proceeding with login");
                    
                    // DELETE OLD SESSIONS
                    $stmt3 = $conn->prepare("DELETE FROM user_sessions WHERE user_id = ?");
                    $stmt3->bind_param("i", $user['id']);
                    $stmt3->execute();
                    $deleted = $stmt3->affected_rows;
                    $stmt3->close();
                    add_debug("Deleted $deleted old sessions");
                    
                    // CREATE PHP SESSION
                    $_SESSION['user_id'] = $user['id'];
                    $_SESSION['username'] = $user['username'];
                    $_SESSION['email'] = $user['email'];
                    $_SESSION['full_name'] = $user['full_name'];
                    $_SESSION['role'] = $user['role'];
                    $_SESSION['license_key'] = $user['license_key'];
                    $_SESSION['login_time'] = time();
                    $_SESSION['last_activity'] = time();
                    add_debug("PHP session created");
                    
                    // RESET LOGIN ATTEMPTS
                    $stmt2 = $conn->prepare("UPDATE users SET login_attempts = 0, locked_until = NULL, last_login = NOW() WHERE id = ?");
                    $stmt2->bind_param("i", $user['id']);
                    $stmt2->execute();
                    $stmt2->close();
                    add_debug("Login attempts reset");
                    
                    // INSERT DATABASE SESSION
                    $session_id = session_id();
                    $ip_address = $_SERVER['REMOTE_ADDR'] ?? 'unknown';
                    $user_agent = substr($_SERVER['HTTP_USER_AGENT'] ?? 'unknown', 0, 255);
                    $expires_at = date('Y-m-d H:i:s', time() + 1800);
                    
                    add_debug("Inserting session: session_id=$session_id, ip=$ip_address");
                    
                    $stmt3 = $conn->prepare("INSERT INTO user_sessions (user_id, session_id, ip_address, user_agent, last_activity, expires_at) VALUES (?, ?, ?, ?, NOW(), ?)");
                    $stmt3->bind_param("issss", $user['id'], $session_id, $ip_address, $user_agent, $expires_at);
                    
                    if ($stmt3->execute()) {
                        $insert_id = $stmt3->insert_id;
                        add_debug("Session inserted successfully! Insert ID: $insert_id");
                        
                        // VERIFY IT WAS INSERTED
                        $verify = $conn->query("SELECT COUNT(*) as count FROM user_sessions WHERE user_id = {$user['id']}");
                        $count = $verify->fetch_assoc()['count'];
                        add_debug("Verification: $count sessions exist for user {$user['id']}");
                    } else {
                        add_debug("ERROR inserting session: " . $stmt3->error);
                        $error = "Session creation failed: " . $stmt3->error;
                    }
                    $stmt3->close();
                    
                    $conn->close();
                    
                    if (empty($error)) {
                        add_debug("Redirecting to index.php");
                        // Show debug before redirect
                        $_SESSION['debug_messages'] = $debug_messages;
                        header('Location: index.php');
                        exit;
                    }
                }
            } else {
                $error = "Invalid password";
                add_debug("Password verification failed");
            }
        }
        
        $stmt->close();
        $conn->close();
    }
}

// Show debug messages if redirected from login
if (isset($_SESSION['debug_messages'])) {
    $debug_messages = array_merge($debug_messages, $_SESSION['debug_messages']);
    unset($_SESSION['debug_messages']);
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Debug Login</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/css/bootstrap.min.css" rel="stylesheet">
    <style>
        body { padding: 20px; background: #f5f5f5; font-family: monospace; }
        .debug-panel {
            background: #1e1e1e;
            color: #00ff00;
            padding: 20px;
            border-radius: 10px;
            margin-bottom: 20px;
            max-height: 300px;
            overflow-y: auto;
        }
        .debug-panel div { margin: 5px 0; }
        .login-form {
            background: white;
            padding: 30px;
            border-radius: 10px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
        }
        .alert { margin-top: 20px; }
    </style>
</head>
<body>
    <div class="container" style="max-width: 600px;">
        <h2>🔍 Debug Login</h2>
        
        <?php if (!empty($debug_messages)): ?>
        <div class="debug-panel">
            <strong>DEBUG LOG:</strong><br>
            <?php foreach ($debug_messages as $msg): ?>
                <div><?php echo htmlspecialchars($msg); ?></div>
            <?php endforeach; ?>
        </div>
        <?php endif; ?>
        
        <div class="login-form">
            <h4>Login Form</h4>
            
            <?php if ($error): ?>
            <div class="alert alert-danger"><?php echo $error; ?></div>
            <?php endif; ?>
            
            <?php if ($show_force_login): ?>
            <div class="alert alert-warning">
                <strong>Session detected!</strong>
                <form method="POST">
                    <input type="hidden" name="username" value="<?php echo htmlspecialchars($_POST['username']); ?>">
                    <input type="hidden" name="password" value="<?php echo htmlspecialchars($_POST['password']); ?>">
                    <input type="hidden" name="force_login" value="1">
                    <button type="submit" class="btn btn-danger mt-2">Force Login</button>
                </form>
            </div>
            <?php else: ?>
            
            <form method="POST">
                <div class="mb-3">
                    <label class="form-label">Username</label>
                    <input type="text" name="username" class="form-control" required>
                </div>
                
                <div class="mb-3">
                    <label class="form-label">Password</label>
                    <input type="password" name="password" class="form-control" required>
                </div>
                
                <button type="submit" class="btn btn-primary">Login</button>
            </form>
            
            <?php endif; ?>
        </div>
        
        <div class="mt-3">
            <a href="diagnostic_single_device.php" class="btn btn-secondary">Run Diagnostic</a>
            <a href="login.php" class="btn btn-secondary">Back to Normal Login</a>
        </div>
        
        <?php if (isset($_SESSION['user_id'])): ?>
        <div class="alert alert-success mt-3">
            ✅ You are logged in as User ID: <?php echo $_SESSION['user_id']; ?>
            <br>
            <a href="index.php">Go to Dashboard</a>
        </div>
        <?php endif; ?>
    </div>
</body>
</html>