<?php
session_start();
require_once 'config.php';
require 'PHPMailer/PHPMailer.php';
require 'PHPMailer/SMTP.php';
require 'PHPMailer/Exception.php';

use PHPMailer\PHPMailer\PHPMailer;
use PHPMailer\PHPMailer\Exception;

// Check if user is logged in
if (!isset($_SESSION['user_id'])) {
    header('Location: login.php');
    exit;
}

$page_title = 'Send Emails';
$user_id = $_SESSION['user_id'];

$conn = getDBConnection();

// Get user's default SMTP credentials
$smtp_result = $conn->query("SELECT * FROM user_smtp_credentials WHERE user_id = $user_id AND is_default = 1 AND is_active = 1 LIMIT 1");

if (!$smtp_result || $smtp_result->num_rows === 0) {
    // No SMTP configured for this user
    $error_message = "No SMTP credentials configured. Please add SMTP credentials in settings first.";
    $show_smtp_btn = true;
} else {
    $smtp_account = $smtp_result->fetch_assoc();
    
    // Check daily limit
    if ($smtp_account['daily_limit'] > 0) {
        // Reset counter if it's a new day
        if ($smtp_account['last_reset_date'] != date('Y-m-d')) {
            $conn->query("UPDATE user_smtp_credentials SET emails_sent_today = 0, last_reset_date = CURDATE() WHERE id = " . $smtp_account['id']);
            $smtp_account['emails_sent_today'] = 0;
        }
        
        if ($smtp_account['emails_sent_today'] >= $smtp_account['daily_limit']) {
            $error_message = "Daily sending limit reached (" . $smtp_account['daily_limit'] . " emails). Try again tomorrow or use a different SMTP account.";
            $show_smtp_btn = true;
        }
    }
    
    if (!isset($error_message)) {
        // Get latest campaign
        $campaign = $conn->query("SELECT * FROM email_campaigns WHERE status != 'completed' ORDER BY id DESC LIMIT 1")->fetch_assoc();
        
        if (!$campaign) {
            $error_message = "No active campaign found. Please compose an email first.";
            $show_compose_btn = true;
        } else {
            // Get pending emails
            $batch_size = defined('BATCH_SIZE') ? BATCH_SIZE : 50;
            $pending = $conn->query("SELECT * FROM email_lists WHERE status = 'pending' LIMIT " . $batch_size);
            
            if ($pending->num_rows == 0) {
                $error_message = "No pending emails to send.";
                $show_upload_btn = true;
            } else {
                $sent = 0;
                $failed = 0;
                $errors = [];
                $limit_reached = false;
                
                while ($row = $pending->fetch_assoc()) {
                    // Check daily limit before each send
                    if ($smtp_account['daily_limit'] > 0 && $smtp_account['emails_sent_today'] >= $smtp_account['daily_limit']) {
                        $limit_reached = true;
                        break;
                    }
                    
                    try {
                        $mail = new PHPMailer(true);
                        
                        // SMTP Configuration from user's credentials
                        $mail->isSMTP();
                        $mail->Host = $smtp_account['smtp_host'];
                        $mail->SMTPAuth = true;
                        $mail->Username = $smtp_account['smtp_username'];
                        $mail->Password = $smtp_account['smtp_password'];
                        $mail->SMTPSecure = $smtp_account['smtp_encryption'];
                        $mail->Port = $smtp_account['smtp_port'];
                        
                        // Sender (from user's SMTP)
                        $mail->setFrom($smtp_account['from_email'], $smtp_account['from_name']);
                        
                        // Recipient
                        $mail->addAddress($row["email"], $row["name"]);
                        
                        // Email content
                        $mail->isHTML(true);
                        $mail->Subject = $campaign["subject"];
                        
                        // Replace personalization tags
                        $body = $campaign["message"];
                        $body = str_replace('{{name}}', htmlspecialchars($row["name"]), $body);
                        $body = str_replace('{{email}}', htmlspecialchars($row["email"]), $body);
                        $mail->Body = $body;
                        
                        $mail->send();
                        
                        // Update status to sent
                        $stmt = $conn->prepare("UPDATE email_lists SET status = 'sent', sent_date = NOW() WHERE id = ?");
                        $stmt->bind_param("i", $row["id"]);
                        $stmt->execute();
                        $stmt->close();
                        
                        // Increment SMTP daily counter
                        $conn->query("UPDATE user_smtp_credentials SET emails_sent_today = emails_sent_today + 1, last_used = NOW() WHERE id = " . $smtp_account['id']);
                        $smtp_account['emails_sent_today']++;
                        
                        $sent++;
                        
                    } catch (Exception $e) {
                        $errors[] = [
                            'email' => $row["email"],
                            'error' => $e->getMessage()
                        ];
                        
                        // Update status to failed
                        $stmt = $conn->prepare("UPDATE email_lists SET status = 'failed', error_message = ? WHERE id = ?");
                        $error = $e->getMessage();
                        $stmt->bind_param("si", $error, $row["id"]);
                        $stmt->execute();
                        $stmt->close();
                        $failed++;
                    }
                }
                
                $success = true;
            }
        }
    }
}

$conn->close();

include 'includes/header.php';
?>

<style>
    body {
        background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    }
    
    .result-card {
        background: white;
        border-radius: 20px;
        box-shadow: 0 20px 60px rgba(0, 0, 0, 0.3);
        overflow: hidden;
        animation: slideUp 0.5s ease-out;
    }
    
    @keyframes slideUp {
        from {
            opacity: 0;
            transform: translateY(30px);
        }
        to {
            opacity: 1;
            transform: translateY(0);
        }
    }
    
    .card-header-gradient {
        padding: 40px 30px;
        text-align: center;
        color: white;
    }
    
    .success-header {
        background: linear-gradient(135deg, var(--success-color) 0%, #059669 100%);
    }
    
    .warning-header {
        background: linear-gradient(135deg, var(--warning-color) 0%, #d97706 100%);
    }
    
    .icon-circle {
        width: 80px;
        height: 80px;
        background: rgba(255, 255, 255, 0.2);
        border-radius: 50%;
        display: flex;
        align-items: center;
        justify-content: center;
        margin: 0 auto 20px;
        animation: pulse 2s infinite;
    }
    
    @keyframes pulse {
        0%, 100% {
            transform: scale(1);
        }
        50% {
            transform: scale(1.05);
        }
    }
    
    .icon-circle i {
        font-size: 40px;
        color: white;
    }
    
    .stats-container {
        display: grid;
        grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
        gap: 20px;
        margin: 30px 0;
    }
    
    .stat-box {
        background: var(--light-bg);
        padding: 25px;
        border-radius: 15px;
        text-align: center;
        transition: transform 0.3s, box-shadow 0.3s;
    }
    
    .stat-box:hover {
        transform: translateY(-5px);
        box-shadow: 0 10px 30px rgba(0, 0, 0, 0.1);
    }
    
    .stat-icon {
        font-size: 32px;
        margin-bottom: 15px;
    }
    
    .stat-number {
        font-size: 36px;
        font-weight: 700;
        margin-bottom: 5px;
    }
    
    .stat-label {
        color: #6b7280;
        font-size: 14px;
        text-transform: uppercase;
        letter-spacing: 1px;
    }
    
    .stat-success {
        color: var(--success-color);
    }
    
    .stat-danger {
        color: var(--danger-color);
    }
    
    .stat-info {
        color: #3b82f6;
    }
    
    .error-list {
        background: #fef2f2;
        border-left: 4px solid var(--danger-color);
        border-radius: 10px;
        padding: 20px;
        margin: 20px 0;
    }
    
    .error-item {
        background: white;
        padding: 15px;
        border-radius: 8px;
        margin-bottom: 10px;
        border: 1px solid #fee2e2;
    }
    
    .error-item:last-child {
        margin-bottom: 0;
    }
    
    .smtp-info-box {
        background: #eff6ff;
        border-left: 4px solid #3b82f6;
        border-radius: 10px;
        padding: 20px;
        margin: 20px 0;
    }
    
    .limit-warning {
        background: #fff7ed;
        border-left: 4px solid #f59e0b;
        border-radius: 10px;
        padding: 20px;
        margin: 20px 0;
    }
    
    .card-body-custom {
        padding: 40px 30px;
    }
    
    .alert-custom {
        border-radius: 12px;
        border: none;
        padding: 20px;
        display: flex;
        align-items: start;
        gap: 15px;
        margin: 20px 0;
    }
    
    .alert-info-custom {
        background: #eff6ff;
        color: #1e40af;
    }
    
    .btn-custom {
        padding: 12px 30px;
        border-radius: 10px;
        font-weight: 600;
        transition: all 0.3s;
    }
    
    .btn-primary-custom {
        background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
        color: white;
        border: none;
    }
    
    .btn-success-custom {
        background: linear-gradient(135deg, #10b981 0%, #059669 100%);
        color: white;
        border: none;
    }
    
    .btn-outline-custom {
        border: 2px solid #667eea;
        color: #667eea;
        background: white;
    }
    
    .btn-custom:hover {
        transform: translateY(-2px);
        box-shadow: 0 5px 20px rgba(0, 0, 0, 0.2);
    }
</style>

<?php if (isset($success)): ?>
    <!-- SUCCESS STATE -->
    <div class="result-card">
        <div class="card-header-gradient success-header">
            <div class="icon-circle">
                <i class="fas fa-check"></i>
            </div>
            <h1 class="mb-2">Batch Sent Successfully!</h1>
            <p class="mb-0 opacity-75">Your email campaign has been processed</p>
        </div>
        
        <div class="card-body-custom">
            <div class="stats-container">
                <div class="stat-box">
                    <div class="stat-icon stat-success">
                        <i class="fas fa-paper-plane"></i>
                    </div>
                    <div class="stat-number stat-success"><?php echo $sent; ?></div>
                    <div class="stat-label">Sent Successfully</div>
                </div>
                
                <div class="stat-box">
                    <div class="stat-icon stat-danger">
                        <i class="fas fa-exclamation-triangle"></i>
                    </div>
                    <div class="stat-number stat-danger"><?php echo $failed; ?></div>
                    <div class="stat-label">Failed</div>
                </div>
                
                <div class="stat-box">
                    <div class="stat-icon stat-info">
                        <i class="fas fa-gauge-high"></i>
                    </div>
                    <div class="stat-number stat-info">
                        <?php echo $smtp_account['emails_sent_today']; ?>
                        <?php if ($smtp_account['daily_limit'] > 0): ?>
                        / <?php echo $smtp_account['daily_limit']; ?>
                        <?php endif; ?>
                    </div>
                    <div class="stat-label">Today's Usage</div>
                </div>
            </div>
            
            <!-- SMTP Info -->
            <div class="smtp-info-box">
                <h6 class="mb-3"><i class="fas fa-server me-2"></i>Sent via SMTP</h6>
                <div class="d-flex justify-content-between flex-wrap gap-3">
                    <div>
                        <small class="text-muted d-block">Account Name</small>
                        <strong><?php echo htmlspecialchars($smtp_account['smtp_name']); ?></strong>
                    </div>
                    <div>
                        <small class="text-muted d-block">From Email</small>
                        <strong><?php echo htmlspecialchars($smtp_account['from_email']); ?></strong>
                    </div>
                    <div>
                        <small class="text-muted d-block">SMTP Host</small>
                        <strong><?php echo htmlspecialchars($smtp_account['smtp_host']); ?></strong>
                    </div>
                </div>
            </div>
            
            <?php if ($limit_reached): ?>
            <div class="limit-warning">
                <div class="d-flex align-items-start gap-3">
                    <i class="fas fa-exclamation-triangle fa-2x"></i>
                    <div>
                        <strong>Daily Limit Reached</strong>
                        <p class="mb-0 mt-2">You've reached your daily sending limit. Remaining emails will be sent tomorrow or you can use a different SMTP account.</p>
                    </div>
                </div>
            </div>
            <?php endif; ?>
            
            <?php if (!empty($errors)): ?>
            <div class="error-list">
                <h5 class="mb-3"><i class="fas fa-exclamation-circle me-2"></i>Error Details</h5>
                <?php foreach (array_slice($errors, 0, 5) as $error): ?>
                <div class="error-item">
                    <div class="fw-bold mb-2">
                        <i class="fas fa-envelope me-2"></i>
                        <?php echo htmlspecialchars($error['email']); ?>
                    </div>
                    <div class="text-muted small">
                        <?php echo htmlspecialchars($error['error']); ?>
                    </div>
                </div>
                <?php endforeach; ?>
                <?php if (count($errors) > 5): ?>
                <p class="text-center mb-0 mt-3"><small>...and <?php echo count($errors) - 5; ?> more errors</small></p>
                <?php endif; ?>
            </div>
            <?php endif; ?>
            
            <div class="alert alert-custom alert-info-custom">
                <i class="fas fa-clock fa-lg"></i>
                <div>
                    <strong>Next Batch Schedule</strong>
                    <p class="mb-0">The next batch will be sent in <?php echo defined('BATCH_INTERVAL') ? BATCH_INTERVAL/60 : 5; ?> minutes</p>
                </div>
            </div>
            
            <div class="d-flex gap-2 justify-content-center flex-wrap">
                <a href="index.php" class="btn btn-primary-custom btn-custom">
                    <i class="fas fa-home me-2"></i>Dashboard
                </a>
                <?php if (!$limit_reached): ?>
                <a href="sender.php" class="btn btn-success-custom btn-custom">
                    <i class="fas fa-redo me-2"></i>Send Next Batch
                </a>
                <?php endif; ?>
                <a href="smtp_settings.php" class="btn btn-outline-custom btn-custom">
                    <i class="fas fa-cog me-2"></i>SMTP Settings
                </a>
            </div>
        </div>
    </div>
    
<?php elseif (isset($error_message)): ?>
    <!-- ERROR/WARNING STATE -->
    <div class="result-card">
        <div class="card-header-gradient warning-header">
            <div class="icon-circle">
                <i class="fas fa-info-circle"></i>
            </div>
            <h1 class="mb-2">No Emails to Send</h1>
            <p class="mb-0 opacity-75">Please check the details below</p>
        </div>
        
        <div class="card-body-custom text-center">
            <div class="alert alert-warning d-flex align-items-center justify-content-center" role="alert">
                <i class="fas fa-exclamation-triangle fa-2x me-3"></i>
                <div class="text-start">
                    <strong>Notice:</strong> <?php echo htmlspecialchars($error_message); ?>
                </div>
            </div>
            
            <div class="alert alert-custom alert-info-custom text-start">
                <i class="fas fa-lightbulb fa-lg"></i>
                <div>
                    <h5 class="mb-3">What to do next?</h5>
                    <ul class="list-unstyled mb-0">
                        <?php if (isset($show_smtp_btn)): ?>
                        <li class="mb-2"><i class="fas fa-chevron-right text-primary me-2"></i>Add SMTP credentials in settings</li>
                        <li class="mb-2"><i class="fas fa-chevron-right text-primary me-2"></i>Make sure it's set as your default SMTP</li>
                        <li class="mb-2"><i class="fas fa-chevron-right text-primary me-2"></i>Test your SMTP connection</li>
                        <?php endif; ?>
                        <?php if (isset($show_compose_btn)): ?>
                        <li class="mb-2"><i class="fas fa-chevron-right text-primary me-2"></i>Create a new email campaign first</li>
                        <?php endif; ?>
                        <?php if (isset($show_upload_btn)): ?>
                        <li class="mb-2"><i class="fas fa-chevron-right text-primary me-2"></i>Upload email addresses from a CSV file</li>
                        <li class="mb-2"><i class="fas fa-chevron-right text-primary me-2"></i>Or reset existing emails to pending status</li>
                        <?php endif; ?>
                    </ul>
                </div>
            </div>
            
            <div class="d-flex gap-2 justify-content-center flex-wrap">
                <?php if (isset($show_smtp_btn)): ?>
                <a href="smtp_settings.php" class="btn btn-success-custom btn-custom">
                    <i class="fas fa-server me-2"></i>SMTP Settings
                </a>
                <?php endif; ?>
                
                <?php if (isset($show_compose_btn)): ?>
                <a href="compose.php" class="btn btn-success-custom btn-custom">
                    <i class="fas fa-edit me-2"></i>Compose Email
                </a>
                <?php endif; ?>
                
                <?php if (isset($show_upload_btn)): ?>
                <a href="upload.php" class="btn btn-success-custom btn-custom">
                    <i class="fas fa-upload me-2"></i>Upload Emails
                </a>
                <a href="check_status.php" class="btn btn-outline-custom btn-custom">
                    <i class="fas fa-sync me-2"></i>Check Status
                </a>
                <?php endif; ?>
                
                <a href="index.php" class="btn btn-primary-custom btn-custom">
                    <i class="fas fa-home me-2"></i>Dashboard
                </a>
            </div>
        </div>
    </div>
<?php endif; ?>

<?php include 'includes/footer.php'; ?>