<?php
$page_title = 'SMTP Settings';
require_once 'includes/header.php';
require_once 'PHPMailer/PHPMailer.php';
require_once 'PHPMailer/SMTP.php';
require_once 'PHPMailer/Exception.php';

use PHPMailer\PHPMailer\PHPMailer;
use PHPMailer\PHPMailer\Exception;

$conn = getDBConnection();
$success = '';
$error = '';

// Handle form submissions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';
    
    if ($action === 'add_smtp') {
        $smtp_name = trim($_POST['smtp_name'] ?? '');
        $smtp_host = trim($_POST['smtp_host'] ?? '');
        $smtp_port = intval($_POST['smtp_port'] ?? 587);
        $smtp_username = trim($_POST['smtp_username'] ?? '');
        $smtp_password = $_POST['smtp_password'] ?? '';
        $smtp_encryption = $_POST['smtp_encryption'] ?? 'tls';
        $from_email = trim($_POST['from_email'] ?? '');
        $from_name = trim($_POST['from_name'] ?? '');
        $daily_limit = intval($_POST['daily_limit'] ?? 0);
        $is_default = isset($_POST['is_default']) ? 1 : 0;
        
        if (empty($smtp_name) || empty($smtp_host) || empty($smtp_username) || empty($from_email)) {
            $error = 'Please fill all required fields';
        } else {
            // If this is set as default, unset other defaults
            if ($is_default) {
                $conn->query("UPDATE user_smtp_credentials SET is_default = 0 WHERE user_id = {$_SESSION['user_id']}");
            }
            
            $stmt = $conn->prepare("INSERT INTO user_smtp_credentials (user_id, smtp_name, smtp_host, smtp_port, smtp_username, smtp_password, smtp_encryption, from_email, from_name, daily_limit, is_default, last_reset_date) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, CURDATE())");
            $stmt->bind_param("issississii", $_SESSION['user_id'], $smtp_name, $smtp_host, $smtp_port, $smtp_username, $smtp_password, $smtp_encryption, $from_email, $from_name, $daily_limit, $is_default);
            
            if ($stmt->execute()) {
                $success = 'SMTP credentials added successfully!';
            } else {
                $error = 'Failed to add SMTP credentials';
            }
            $stmt->close();
        }
    }
    elseif ($action === 'test_smtp') {
        $smtp_id = intval($_POST['smtp_id'] ?? 0);
        
        $stmt = $conn->prepare("SELECT * FROM user_smtp_credentials WHERE id = ? AND user_id = ?");
        $stmt->bind_param("ii", $smtp_id, $_SESSION['user_id']);
        $stmt->execute();
        $smtp = $stmt->get_result()->fetch_assoc();
        $stmt->close();
        
        if ($smtp) {
            $mail = new PHPMailer(true);
            
            try {
                $mail->isSMTP();
                $mail->Host = $smtp['smtp_host'];
                $mail->SMTPAuth = true;
                $mail->Username = $smtp['smtp_username'];
                $mail->Password = $smtp['smtp_password'];
                $mail->SMTPSecure = $smtp['smtp_encryption'];
                $mail->Port = $smtp['smtp_port'];
                $mail->setFrom($smtp['from_email'], $smtp['from_name']);
                $mail->addAddress($smtp['from_email']); // Send test to self
                
                $mail->isHTML(true);
                $mail->Subject = 'SMTP Test - Email Sender';
                $mail->Body = '<h2>SMTP Test Successful!</h2><p>Your SMTP credentials are working correctly.</p><p>SMTP Server: ' . $smtp['smtp_host'] . '</p>';
                $mail->AltBody = 'SMTP Test Successful! Your SMTP credentials are working correctly.';
                
                $mail->send();
                
                // Update status
                $conn->query("UPDATE user_smtp_credentials SET status = 'verified', test_result = 'Test email sent successfully', last_used = NOW() WHERE id = $smtp_id");
                
                $success = 'Test email sent successfully! Check your inbox at ' . $smtp['from_email'];
            } catch (Exception $e) {
                $test_result = $mail->ErrorInfo;
                $conn->query("UPDATE user_smtp_credentials SET status = 'failed', test_result = '" . $conn->real_escape_string($test_result) . "' WHERE id = $smtp_id");
                $error = 'SMTP test failed: ' . $test_result;
            }
        }
    }
    elseif ($action === 'set_default') {
        $smtp_id = intval($_POST['smtp_id'] ?? 0);
        
        // Unset all defaults
        $conn->query("UPDATE user_smtp_credentials SET is_default = 0 WHERE user_id = {$_SESSION['user_id']}");
        
        // Set new default
        $stmt = $conn->prepare("UPDATE user_smtp_credentials SET is_default = 1 WHERE id = ? AND user_id = ?");
        $stmt->bind_param("ii", $smtp_id, $_SESSION['user_id']);
        $stmt->execute();
        $stmt->close();
        
        $success = 'Default SMTP updated';
    }
    elseif ($action === 'edit_smtp') {
        $smtp_id = intval($_POST['smtp_id'] ?? 0);
        $smtp_name = trim($_POST['smtp_name'] ?? '');
        $smtp_host = trim($_POST['smtp_host'] ?? '');
        $smtp_port = intval($_POST['smtp_port'] ?? 587);
        $smtp_username = trim($_POST['smtp_username'] ?? '');
        $smtp_password = $_POST['smtp_password'] ?? '';
        $smtp_encryption = $_POST['smtp_encryption'] ?? 'tls';
        $from_email = trim($_POST['from_email'] ?? '');
        $from_name = trim($_POST['from_name'] ?? '');
        $daily_limit = intval($_POST['daily_limit'] ?? 0);
        
        if (empty($smtp_name) || empty($smtp_host) || empty($smtp_username) || empty($from_email)) {
            $error = 'Please fill all required fields';
        } else {
            // Only update password if a new one is provided
            if (!empty($smtp_password)) {
                $stmt = $conn->prepare("UPDATE user_smtp_credentials SET smtp_name = ?, smtp_host = ?, smtp_port = ?, smtp_username = ?, smtp_password = ?, smtp_encryption = ?, from_email = ?, from_name = ?, daily_limit = ? WHERE id = ? AND user_id = ?");
                $stmt->bind_param("ssisssssiiii", $smtp_name, $smtp_host, $smtp_port, $smtp_username, $smtp_password, $smtp_encryption, $from_email, $from_name, $daily_limit, $smtp_id, $_SESSION['user_id']);
            } else {
                $stmt = $conn->prepare("UPDATE user_smtp_credentials SET smtp_name = ?, smtp_host = ?, smtp_port = ?, smtp_username = ?, smtp_encryption = ?, from_email = ?, from_name = ?, daily_limit = ? WHERE id = ? AND user_id = ?");
                $stmt->bind_param("ssissssiii", $smtp_name, $smtp_host, $smtp_port, $smtp_username, $smtp_encryption, $from_email, $from_name, $daily_limit, $smtp_id, $_SESSION['user_id']);
            }
            
            if ($stmt->execute()) {
                $success = 'SMTP credentials updated successfully!';
            } else {
                $error = 'Failed to update SMTP credentials';
            }
            $stmt->close();
        }
    }
    elseif ($action === 'delete_smtp') {
        $smtp_id = intval($_POST['smtp_id'] ?? 0);
        
        $stmt = $conn->prepare("DELETE FROM user_smtp_credentials WHERE id = ? AND user_id = ?");
        $stmt->bind_param("ii", $smtp_id, $_SESSION['user_id']);
        $stmt->execute();
        $stmt->close();
        
        $success = 'SMTP credentials deleted';
    }
    elseif ($action === 'toggle_status') {
        $smtp_id = intval($_POST['smtp_id'] ?? 0);
        $new_status = intval($_POST['new_status'] ?? 0);
        
        $stmt = $conn->prepare("UPDATE user_smtp_credentials SET is_active = ? WHERE id = ? AND user_id = ?");
        $stmt->bind_param("iii", $new_status, $smtp_id, $_SESSION['user_id']);
        $stmt->execute();
        $stmt->close();
        
        $success = $new_status ? 'SMTP enabled' : 'SMTP disabled';
    }
}

// Get user's SMTP credentials
$smtp_list = $conn->query("SELECT * FROM user_smtp_credentials WHERE user_id = {$_SESSION['user_id']} ORDER BY is_default DESC, created_date DESC");

// Get statistics
$stats = $conn->query("SELECT COUNT(*) as total, SUM(is_active) as active, SUM(is_default) as has_default FROM user_smtp_credentials WHERE user_id = {$_SESSION['user_id']}")->fetch_assoc();

$conn->close();
?>

<style>
    .smtp-card {
        background: white;
        border-radius: 12px;
        padding: 20px;
        margin-bottom: 20px;
        box-shadow: 0 2px 8px rgba(0,0,0,0.1);
        transition: all 0.3s;
    }
    
    .smtp-card:hover {
        box-shadow: 0 4px 12px rgba(0,0,0,0.15);
    }
    
    .smtp-header {
        display: flex;
        justify-content: space-between;
        align-items: center;
        margin-bottom: 15px;
        padding-bottom: 15px;
        border-bottom: 2px solid #e5e7eb;
    }
    
    .smtp-name {
        font-size: 1.25rem;
        font-weight: 700;
        color: #1f2937;
    }
    
    .smtp-badges {
        display: flex;
        gap: 8px;
    }
    
    .smtp-info {
        display: grid;
        grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
        gap: 15px;
        margin-bottom: 15px;
    }
    
    .info-item {
        display: flex;
        flex-direction: column;
    }
    
    .info-label {
        font-size: 0.75rem;
        color: #6b7280;
        font-weight: 600;
        text-transform: uppercase;
        margin-bottom: 4px;
    }
    
    .info-value {
        font-size: 0.95rem;
        color: #1f2937;
        font-weight: 500;
    }
    
    .smtp-actions {
        display: flex;
        gap: 10px;
        flex-wrap: wrap;
    }
    
    .status-verified {
        background: #d1fae5;
        color: #065f46;
        padding: 4px 12px;
        border-radius: 12px;
        font-size: 0.75rem;
        font-weight: 600;
    }
    
    .status-pending {
        background: #fef3c7;
        color: #92400e;
        padding: 4px 12px;
        border-radius: 12px;
        font-size: 0.75rem;
        font-weight: 600;
    }
    
    .status-failed {
        background: #fee2e2;
        color: #991b1b;
        padding: 4px 12px;
        border-radius: 12px;
        font-size: 0.75rem;
        font-weight: 600;
    }
    
    .card-custom {
        background: white;
        border-radius: 15px;
        box-shadow: 0 2px 10px rgba(0,0,0,0.1);
        margin-bottom: 30px;
    }
    
    .card-header-custom {
        background: linear-gradient(135deg, var(--primary-color) 0%, #764ba2 100%);
        color: white;
        padding: 20px;
        border-radius: 15px 15px 0 0;
        font-weight: 700;
        font-size: 1.1rem;
    }
    
    .card-body-custom {
        padding: 25px;
    }
    
    .form-row {
        display: grid;
        grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
        gap: 15px;
        margin-bottom: 20px;
    }
    
    .form-label-custom {
        display: block;
        font-weight: 600;
        color: #374151;
        margin-bottom: 8px;
        font-size: 0.9rem;
    }
    
    .form-control-custom {
        width: 100%;
        padding: 12px 15px;
        border: 2px solid #e5e7eb;
        border-radius: 10px;
        transition: all 0.3s;
        font-size: 0.95rem;
    }
    
    .form-control-custom:focus {
        outline: none;
        border-color: var(--primary-color);
        box-shadow: 0 0 0 3px rgba(102, 126, 234, 0.1);
    }
    
    .btn-custom {
        padding: 12px 24px;
        border-radius: 10px;
        border: none;
        font-weight: 600;
        transition: all 0.3s;
        cursor: pointer;
    }
    
    .btn-primary-custom {
        background: linear-gradient(135deg, var(--primary-color) 0%, #764ba2 100%);
        color: white;
    }
    
    .btn-primary-custom:hover {
        transform: translateY(-2px);
        box-shadow: 0 5px 15px rgba(102, 126, 234, 0.3);
    }
    
    .alert-info-custom {
        background: #eff6ff;
        border-left: 4px solid #3b82f6;
        padding: 15px;
        border-radius: 8px;
        margin-bottom: 20px;
    }
    
    .stats-grid {
        display: grid;
        grid-template-columns: repeat(auto-fit, minmax(150px, 1fr));
        gap: 15px;
        margin-bottom: 25px;
    }
    
    .stat-card {
        background: linear-gradient(135deg, var(--primary-color) 0%, #764ba2 100%);
        color: white;
        padding: 20px;
        border-radius: 12px;
        text-align: center;
    }
    
    .stat-number {
        font-size: 2rem;
        font-weight: 700;
        margin-bottom: 5px;
    }
    
    .stat-label {
        font-size: 0.85rem;
        opacity: 0.9;
    }
    
    .provider-quick {
        display: grid;
        grid-template-columns: repeat(auto-fit, minmax(120px, 1fr));
        gap: 10px;
        margin-bottom: 25px;
    }
    
    .provider-btn {
        padding: 15px 10px;
        border: 2px solid #e5e7eb;
        border-radius: 10px;
        background: white;
        cursor: pointer;
        transition: all 0.3s;
        text-align: center;
    }
    
    .provider-btn:hover {
        border-color: var(--primary-color);
        background: rgba(102, 126, 234, 0.05);
    }
    
    .provider-icon {
        font-size: 2rem;
        margin-bottom: 8px;
    }
</style>

<div class="container-custom">
    <div class="page-header">
        <h1><i class="fas fa-server me-3"></i><?php echo $page_title; ?></h1>
        <p>Manage your SMTP credentials for sending emails</p>
    </div>
    
    <?php if ($success): ?>
    <div class="alert alert-success alert-dismissible fade show">
        <i class="fas fa-check-circle me-2"></i><?php echo htmlspecialchars($success); ?>
        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
    </div>
    <?php endif; ?>
    
    <?php if ($error): ?>
    <div class="alert alert-danger alert-dismissible fade show">
        <i class="fas fa-exclamation-circle me-2"></i><?php echo htmlspecialchars($error); ?>
        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
    </div>
    <?php endif; ?>
    
    <!-- Statistics -->
    <div class="stats-grid">
        <div class="stat-card">
            <div class="stat-number"><?php echo $stats['total'] ?? 0; ?></div>
            <div class="stat-label">Total SMTP Accounts</div>
        </div>
        <div class="stat-card">
            <div class="stat-number"><?php echo $stats['active'] ?? 0; ?></div>
            <div class="stat-label">Active</div>
        </div>
        <div class="stat-card">
            <div class="stat-number"><?php echo $stats['has_default'] ?? 0; ?></div>
            <div class="stat-label">Default Set</div>
        </div>
    </div>
    
    <!-- Add New SMTP -->
    <div class="card-custom">
        <div class="card-header-custom">
            <i class="fas fa-plus-circle me-2"></i>
            Add New SMTP Credentials
        </div>
        <div class="card-body-custom">
            <div class="alert-info-custom">
                <i class="fas fa-info-circle me-2"></i>
                <strong>Quick Setup:</strong> Click a provider below to auto-fill common settings
            </div>
            
            <div class="provider-quick">
                <div class="provider-btn" onclick="fillGmail()">
                    <div class="provider-icon"><i class="fab fa-google"></i></div>
                    <div>Gmail</div>
                </div>
                <div class="provider-btn" onclick="fillOutlook()">
                    <div class="provider-icon"><i class="fab fa-microsoft"></i></div>
                    <div>Outlook</div>
                </div>
                <div class="provider-btn" onclick="fillYahoo()">
                    <div class="provider-icon"><i class="fab fa-yahoo"></i></div>
                    <div>Yahoo</div>
                </div>
                <div class="provider-btn" onclick="fillSendGrid()">
                    <div class="provider-icon"><i class="fas fa-paper-plane"></i></div>
                    <div>SendGrid</div>
                </div>
            </div>
            
            <form method="POST">
                <input type="hidden" name="action" value="add_smtp">
                
                <div class="mb-3">
                    <label class="form-label-custom">SMTP Name *</label>
                    <input type="text" name="smtp_name" class="form-control-custom" placeholder="My Gmail SMTP" required>
                </div>
                
                <div class="form-row">
                    <div>
                        <label class="form-label-custom">SMTP Host *</label>
                        <input type="text" name="smtp_host" id="smtp_host" class="form-control-custom" placeholder="smtp.gmail.com" required>
                    </div>
                    
                    <div>
                        <label class="form-label-custom">Port *</label>
                        <input type="number" name="smtp_port" id="smtp_port" class="form-control-custom" value="587" required>
                    </div>
                    
                    <div>
                        <label class="form-label-custom">Encryption</label>
                        <select name="smtp_encryption" id="smtp_encryption" class="form-control-custom">
                            <option value="tls">TLS (Recommended)</option>
                            <option value="ssl">SSL</option>
                            <option value="none">None</option>
                        </select>
                    </div>
                </div>
                
                <div class="form-row">
                    <div>
                        <label class="form-label-custom">SMTP Username *</label>
                        <input type="text" name="smtp_username" id="smtp_username" class="form-control-custom" placeholder="your-email@gmail.com" required>
                    </div>
                    
                    <div>
                        <label class="form-label-custom">SMTP Password *</label>
                        <input type="password" name="smtp_password" class="form-control-custom" placeholder="Your password or app password" required>
                    </div>
                </div>
                
                <div class="form-row">
                    <div>
                        <label class="form-label-custom">From Email *</label>
                        <input type="email" name="from_email" id="from_email" class="form-control-custom" placeholder="sender@yourdomain.com" required>
                    </div>
                    
                    <div>
                        <label class="form-label-custom">From Name *</label>
                        <input type="text" name="from_name" class="form-control-custom" placeholder="Your Name or Company" required>
                    </div>
                    
                    <div>
                        <label class="form-label-custom">Daily Limit (0 = unlimited)</label>
                        <input type="number" name="daily_limit" class="form-control-custom" value="0" min="0">
                    </div>
                </div>
                
                <div class="mb-3">
                    <label class="form-check-label">
                        <input type="checkbox" name="is_default" class="form-check-input">
                        <strong>Set as default SMTP</strong>
                    </label>
                </div>
                
                <button type="submit" class="btn-custom btn-primary-custom">
                    <i class="fas fa-plus-circle me-2"></i>
                    Add SMTP Credentials
                </button>
            </form>
        </div>
    </div>
    
    <!-- SMTP List -->
    <h4 class="mb-3">
        <i class="fas fa-list me-2"></i>
        Your SMTP Credentials
    </h4>
    
    <?php if ($smtp_list->num_rows == 0): ?>
    <div class="alert-info-custom">
        <i class="fas fa-info-circle"></i>
        No SMTP credentials added yet. Add your first SMTP above to start sending emails with your own credentials.
    </div>
    <?php else: ?>
    
    <?php while ($smtp = $smtp_list->fetch_assoc()): ?>
    <div class="smtp-card">
        <div class="smtp-header">
            <div class="smtp-name">
                <i class="fas fa-server me-2"></i>
                <?php echo htmlspecialchars($smtp['smtp_name']); ?>
            </div>
            <div class="smtp-badges">
                <?php if ($smtp['is_default']): ?>
                    <span class="badge bg-warning">Default</span>
                <?php endif; ?>
                
                <?php if ($smtp['is_active']): ?>
                    <span class="badge bg-success">Active</span>
                <?php else: ?>
                    <span class="badge bg-secondary">Inactive</span>
                <?php endif; ?>
                
                <span class="badge status-<?php echo $smtp['status']; ?>">
                    <?php echo ucfirst($smtp['status']); ?>
                </span>
            </div>
        </div>
        
        <div class="smtp-info">
            <div class="info-item">
                <div class="info-label">SMTP Host</div>
                <div class="info-value"><?php echo htmlspecialchars($smtp['smtp_host']); ?>:<?php echo $smtp['smtp_port']; ?></div>
            </div>
            
            <div class="info-item">
                <div class="info-label">Username</div>
                <div class="info-value"><?php echo htmlspecialchars($smtp['smtp_username']); ?></div>
            </div>
            
            <div class="info-item">
                <div class="info-label">From Email</div>
                <div class="info-value"><?php echo htmlspecialchars($smtp['from_email']); ?></div>
            </div>
            
            <div class="info-item">
                <div class="info-label">Encryption</div>
                <div class="info-value"><?php echo strtoupper($smtp['smtp_encryption']); ?></div>
            </div>
            
            <div class="info-item">
                <div class="info-label">Daily Limit</div>
                <div class="info-value"><?php echo $smtp['daily_limit'] == 0 ? 'Unlimited' : $smtp['daily_limit']; ?></div>
            </div>
            
            <div class="info-item">
                <div class="info-label">Sent Today</div>
                <div class="info-value"><?php echo $smtp['emails_sent_today']; ?> emails</div>
            </div>
        </div>
        
        <?php if ($smtp['test_result']): ?>
        <div class="alert-info-custom mb-3">
            <small><strong>Last Test:</strong> <?php echo htmlspecialchars($smtp['test_result']); ?></small>
        </div>
        <?php endif; ?>
        
        <div class="smtp-actions">
            <form method="POST" style="display:inline;">
                <input type="hidden" name="action" value="test_smtp">
                <input type="hidden" name="smtp_id" value="<?php echo $smtp['id']; ?>">
                <button type="submit" class="btn btn-sm btn-info">
                    <i class="fas fa-vial me-1"></i> Test
                </button>
            </form>
            
            <button class="btn btn-sm btn-primary" data-bs-toggle="modal" data-bs-target="#editModal<?php echo $smtp['id']; ?>">
                <i class="fas fa-edit me-1"></i> Edit
            </button>
            
            <?php if (!$smtp['is_default']): ?>
            <form method="POST" style="display:inline;">
                <input type="hidden" name="action" value="set_default">
                <input type="hidden" name="smtp_id" value="<?php echo $smtp['id']; ?>">
                <button type="submit" class="btn btn-sm btn-warning">
                    <i class="fas fa-star me-1"></i> Set Default
                </button>
            </form>
            <?php endif; ?>
            
            <form method="POST" style="display:inline;">
                <input type="hidden" name="action" value="toggle_status">
                <input type="hidden" name="smtp_id" value="<?php echo $smtp['id']; ?>">
                <input type="hidden" name="new_status" value="<?php echo $smtp['is_active'] ? 0 : 1; ?>">
                <button type="submit" class="btn btn-sm <?php echo $smtp['is_active'] ? 'btn-secondary' : 'btn-success'; ?>">
                    <i class="fas fa-power-off me-1"></i> <?php echo $smtp['is_active'] ? 'Disable' : 'Enable'; ?>
                </button>
            </form>
            
            <form method="POST" style="display:inline;" onsubmit="return confirm('Delete this SMTP?');">
                <input type="hidden" name="action" value="delete_smtp">
                <input type="hidden" name="smtp_id" value="<?php echo $smtp['id']; ?>">
                <button type="submit" class="btn btn-sm btn-danger">
                    <i class="fas fa-trash me-1"></i> Delete
                </button>
            </form>
        </div>
    </div>
    
    <!-- Edit Modal for this SMTP -->
    <div class="modal fade" id="editModal<?php echo $smtp['id']; ?>" tabindex="-1" aria-hidden="true">
        <div class="modal-dialog modal-lg">
            <div class="modal-content">
                <div class="modal-header" style="background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); color: white;">
                    <h5 class="modal-title">
                        <i class="fas fa-edit me-2"></i>
                        Edit SMTP: <?php echo htmlspecialchars($smtp['smtp_name']); ?>
                    </h5>
                    <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal"></button>
                </div>
                <form method="POST">
                    <div class="modal-body">
                        <input type="hidden" name="action" value="edit_smtp">
                        <input type="hidden" name="smtp_id" value="<?php echo $smtp['id']; ?>">
                        
                        <div class="mb-3">
                            <label class="form-label-custom">SMTP Name *</label>
                            <input type="text" name="smtp_name" class="form-control-custom" value="<?php echo htmlspecialchars($smtp['smtp_name']); ?>" required>
                        </div>
                        
                        <div class="row">
                            <div class="col-md-8">
                                <div class="mb-3">
                                    <label class="form-label-custom">SMTP Host *</label>
                                    <input type="text" name="smtp_host" class="form-control-custom" value="<?php echo htmlspecialchars($smtp['smtp_host']); ?>" required>
                                </div>
                            </div>
                            
                            <div class="col-md-4">
                                <div class="mb-3">
                                    <label class="form-label-custom">Port *</label>
                                    <input type="number" name="smtp_port" class="form-control-custom" value="<?php echo $smtp['smtp_port']; ?>" required>
                                </div>
                            </div>
                        </div>
                        
                        <div class="row">
                            <div class="col-md-6">
                                <div class="mb-3">
                                    <label class="form-label-custom">SMTP Username *</label>
                                    <input type="text" name="smtp_username" class="form-control-custom" value="<?php echo htmlspecialchars($smtp['smtp_username']); ?>" required>
                                </div>
                            </div>
                            
                            <div class="col-md-6">
                                <div class="mb-3">
                                    <label class="form-label-custom">Encryption</label>
                                    <select name="smtp_encryption" class="form-control-custom">
                                        <option value="tls" <?php echo $smtp['smtp_encryption'] == 'tls' ? 'selected' : ''; ?>>TLS</option>
                                        <option value="ssl" <?php echo $smtp['smtp_encryption'] == 'ssl' ? 'selected' : ''; ?>>SSL</option>
                                        <option value="none" <?php echo $smtp['smtp_encryption'] == 'none' ? 'selected' : ''; ?>>None</option>
                                    </select>
                                </div>
                            </div>
                        </div>
                        
                        <div class="mb-3">
                            <label class="form-label-custom">SMTP Password</label>
                            <input type="password" name="smtp_password" class="form-control-custom" placeholder="Leave blank to keep current password">
                            <small class="text-muted">Only enter a new password if you want to change it</small>
                        </div>
                        
                        <div class="row">
                            <div class="col-md-6">
                                <div class="mb-3">
                                    <label class="form-label-custom">From Email *</label>
                                    <input type="email" name="from_email" class="form-control-custom" value="<?php echo htmlspecialchars($smtp['from_email']); ?>" required>
                                </div>
                            </div>
                            
                            <div class="col-md-6">
                                <div class="mb-3">
                                    <label class="form-label-custom">From Name *</label>
                                    <input type="text" name="from_name" class="form-control-custom" value="<?php echo htmlspecialchars($smtp['from_name']); ?>" required>
                                </div>
                            </div>
                        </div>
                        
                        <div class="mb-3">
                            <label class="form-label-custom">Daily Limit (0 = unlimited)</label>
                            <input type="number" name="daily_limit" class="form-control-custom" value="<?php echo $smtp['daily_limit']; ?>" min="0">
                        </div>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                        <button type="submit" class="btn btn-primary">
                            <i class="fas fa-save me-2"></i>Save Changes
                        </button>
                    </div>
                </form>
            </div>
        </div>
    </div>
    <?php endwhile; ?>
    
    <?php endif; ?>
    
    <!-- Help Section -->
    <div class="card-custom mt-4">
        <div class="card-header-custom">
            <i class="fas fa-question-circle me-2"></i>
            Common SMTP Providers
        </div>
        <div class="card-body-custom">
            <div class="row">
                <div class="col-md-6">
                    <h6><i class="fab fa-google me-2"></i>Gmail</h6>
                    <ul class="small">
                        <li>Host: smtp.gmail.com</li>
                        <li>Port: 587 (TLS)</li>
                        <li>Note: Use App Password, not regular password</li>
                    </ul>
                </div>
                
                <div class="col-md-6">
                    <h6><i class="fab fa-microsoft me-2"></i>Outlook/Office365</h6>
                    <ul class="small">
                        <li>Host: smtp.office365.com</li>
                        <li>Port: 587 (TLS)</li>
                        <li>Username: Your full email address</li>
                    </ul>
                </div>
                
                <div class="col-md-6">
                    <h6><i class="fas fa-envelope me-2"></i>Yahoo Mail</h6>
                    <ul class="small">
                        <li>Host: smtp.mail.yahoo.com</li>
                        <li>Port: 587 (TLS) or 465 (SSL)</li>
                        <li>Note: Generate App Password in settings</li>
                    </ul>
                </div>
                
                <div class="col-md-6">
                    <h6><i class="fas fa-server me-2"></i>SendGrid</h6>
                    <ul class="small">
                        <li>Host: smtp.sendgrid.net</li>
                        <li>Port: 587 (TLS)</li>
                        <li>Username: apikey</li>
                        <li>Password: Your SendGrid API Key</li>
                    </ul>
                </div>
            </div>
        </div>
    </div>
</div>

<script>
function fillGmail() {
    document.getElementById('smtp_host').value = 'smtp.gmail.com';
    document.getElementById('smtp_port').value = '587';
    document.getElementById('smtp_encryption').value = 'tls';
    document.getElementById('smtp_username').placeholder = 'your-email@gmail.com';
    document.getElementById('from_email').placeholder = 'your-email@gmail.com';
}

function fillOutlook() {
    document.getElementById('smtp_host').value = 'smtp.office365.com';
    document.getElementById('smtp_port').value = '587';
    document.getElementById('smtp_encryption').value = 'tls';
    document.getElementById('smtp_username').placeholder = 'your-email@outlook.com';
    document.getElementById('from_email').placeholder = 'your-email@outlook.com';
}

function fillYahoo() {
    document.getElementById('smtp_host').value = 'smtp.mail.yahoo.com';
    document.getElementById('smtp_port').value = '587';
    document.getElementById('smtp_encryption').value = 'tls';
    document.getElementById('smtp_username').placeholder = 'your-email@yahoo.com';
    document.getElementById('from_email').placeholder = 'your-email@yahoo.com';
}

function fillSendGrid() {
    document.getElementById('smtp_host').value = 'smtp.sendgrid.net';
    document.getElementById('smtp_port').value = '587';
    document.getElementById('smtp_encryption').value = 'tls';
    document.getElementById('smtp_username').value = 'apikey';
}
</script>

<?php require_once 'includes/footer.php'; ?>