<?php
session_start();
require_once 'config.php';

$page_title = 'Upload Email List';

if ($_SERVER["REQUEST_METHOD"] == "POST" && isset($_FILES["csvfile"])) {
    $conn = getDBConnection();
    $file = $_FILES["csvfile"]["tmp_name"];
    
    // Check if file was uploaded successfully
    if (!is_uploaded_file($file)) {
        $_SESSION["error"] = "File upload failed.";
        header("Location: upload.php");
        exit;
    }
    
    $handle = fopen($file, "r");
    if ($handle === FALSE) {
        $_SESSION["error"] = "Could not open CSV file.";
        header("Location: upload.php");
        exit;
    }
    
    $imported = 0;
    $duplicates = 0;
    $invalid = 0;
    
    // Skip header row
    fgetcsv($handle);
    
    while (($data = fgetcsv($handle)) !== FALSE) {
        // Skip empty rows
        if (empty($data[0])) continue;
        
        $email = trim($data[0]);
        $name = isset($data[1]) ? trim($data[1]) : "";
        
        // Validate email
        if (filter_var($email, FILTER_VALIDATE_EMAIL)) {
            // Insert with 'pending' status
            $stmt = $conn->prepare("INSERT IGNORE INTO email_lists (email, name, status) VALUES (?, ?, 'pending')");
            $stmt->bind_param("ss", $email, $name);
            
            if ($stmt->execute() && $stmt->affected_rows > 0) {
                $imported++;
            } else {
                $duplicates++;
            }
            $stmt->close();
        } else {
            $invalid++;
        }
    }
    
    fclose($handle);
    $conn->close();
    
    $_SESSION["imported"] = $imported;
    $_SESSION["duplicates"] = $duplicates;
    $_SESSION["invalid"] = $invalid;
    
    header("Location: upload.php");
    exit;
}

include 'includes/header.php';
?>

<style>
    .upload-area {
        border: 2px dashed var(--border-color);
        border-radius: 12px;
        padding: 3rem;
        text-align: center;
        background: var(--light-bg);
        transition: all 0.3s;
        cursor: pointer;
    }
    
    .upload-area:hover {
        border-color: var(--primary-color);
        background: rgba(79, 70, 229, 0.05);
    }
    
    .upload-icon {
        font-size: 4rem;
        color: var(--primary-color);
        margin-bottom: 1rem;
    }
    
    .info-card {
        background: linear-gradient(135deg, #dbeafe 0%, #e0e7ff 100%);
        border-radius: 12px;
        padding: 1.5rem;
        border-left: 4px solid var(--info-color);
    }
    
    .sample-csv {
        background: white;
        border: 1px solid var(--border-color);
        border-radius: 8px;
        padding: 1rem;
        font-family: 'Courier New', monospace;
        font-size: 0.875rem;
        margin-top: 1rem;
    }
    
    .result-card {
        background: white;
        border-radius: 12px;
        padding: 1.5rem;
        box-shadow: var(--shadow-md);
        margin-bottom: 1.5rem;
    }
    
    .result-item {
        display: flex;
        align-items: center;
        justify-content: space-between;
        padding: 1rem;
        background: var(--light-bg);
        border-radius: 8px;
        margin-bottom: 0.75rem;
    }
    
    .result-item:last-child {
        margin-bottom: 0;
    }
    
    .result-label {
        display: flex;
        align-items: center;
        gap: 0.75rem;
        font-weight: 600;
    }
    
    .result-badge {
        font-size: 1.5rem;
        font-weight: 700;
        padding: 0.5rem 1rem;
        border-radius: 8px;
    }
    
    .badge-success {
        background: rgba(16, 185, 129, 0.1);
        color: var(--success-color);
    }
    
    .badge-warning {
        background: rgba(245, 158, 11, 0.1);
        color: var(--warning-color);
    }
    
    .badge-danger {
        background: rgba(239, 68, 68, 0.1);
        color: var(--danger-color);
    }
</style>

<div class="page-header">
    <h1 class="page-title">
        <i class="fas fa-upload"></i>
        Upload Email List
    </h1>
    <p class="page-subtitle">Import your email recipients from a CSV file</p>
</div>

<?php if (isset($_SESSION["imported"])): ?>
<div class="result-card">
    <h5 class="mb-3"><i class="fas fa-check-circle text-success me-2"></i>Upload Complete!</h5>
    
    <div class="result-item">
        <div class="result-label">
            <i class="fas fa-check-circle text-success fa-lg"></i>
            <span>Successfully Imported</span>
        </div>
        <span class="result-badge badge-success"><?php echo $_SESSION['imported']; ?></span>
    </div>
    
    <?php if ($_SESSION['duplicates'] > 0): ?>
    <div class="result-item">
        <div class="result-label">
            <i class="fas fa-copy text-warning fa-lg"></i>
            <span>Duplicates Skipped</span>
        </div>
        <span class="result-badge badge-warning"><?php echo $_SESSION['duplicates']; ?></span>
    </div>
    <?php endif; ?>
    
    <?php if ($_SESSION['invalid'] > 0): ?>
    <div class="result-item">
        <div class="result-label">
            <i class="fas fa-exclamation-triangle text-danger fa-lg"></i>
            <span>Invalid Emails</span>
        </div>
        <span class="result-badge badge-danger"><?php echo $_SESSION['invalid']; ?></span>
    </div>
    <?php endif; ?>
    
    <div class="mt-3 d-flex gap-2">
        <a href="list.php" class="btn btn-primary-custom btn-custom">
            <i class="fas fa-list"></i>
            View Email List
        </a>
        <a href="sender.php" class="btn btn-success-custom btn-custom">
            <i class="fas fa-paper-plane"></i>
            Start Sending
        </a>
    </div>
</div>
<?php 
    unset($_SESSION['imported']);
    unset($_SESSION['duplicates']);
    unset($_SESSION['invalid']);
endif; 
?>

<?php if (isset($_SESSION["error"])): ?>
<div class="alert alert-custom alert-danger-custom alert-dismissible fade show" role="alert">
    <i class="fas fa-exclamation-circle fa-lg"></i>
    <div><?php echo htmlspecialchars($_SESSION["error"]); ?></div>
    <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
</div>
<?php unset($_SESSION["error"]); endif; ?>

<div class="row">
    <div class="col-lg-8">
        <div class="card-custom">
            <div class="card-body-custom">
                <form method="POST" enctype="multipart/form-data" id="uploadForm">
                    <div class="upload-area" onclick="document.getElementById('csvfile').click()">
                        <input type="file" name="csvfile" id="csvfile" accept=".csv" required style="display: none;" onchange="displayFileName()">
                        <div class="upload-icon">
                            <i class="fas fa-cloud-upload-alt"></i>
                        </div>
                        <h4>Click to Upload CSV File</h4>
                        <p class="text-muted mb-0">or drag and drop your file here</p>
                        <p class="text-muted mt-2" id="fileName"></p>
                    </div>
                    
                    <div class="mt-4 text-center">
                        <button type="submit" class="btn btn-primary-custom btn-custom btn-lg">
                            <i class="fas fa-upload"></i>
                            Upload and Import
                        </button>
                    </div>
                </form>
            </div>
        </div>
    </div>
    
    <div class="col-lg-4">
        <div class="info-card">
            <h5 class="mb-3">
                <i class="fas fa-info-circle me-2"></i>
                CSV Format Instructions
            </h5>
            <ul class="mb-3" style="padding-left: 1.25rem;">
                <li class="mb-2">First column: Email address (required)</li>
                <li class="mb-2">Second column: Name (optional)</li>
                <li class="mb-2">First row will be treated as header and skipped</li>
            </ul>
            
            <h6 class="mb-2"><strong>Sample CSV:</strong></h6>
            <div class="sample-csv">
                email,name<br>
                john@example.com,John Doe<br>
                jane@example.com,Jane Smith<br>
                bob@example.com,Bob Johnson
            </div>
            
            <div class="mt-3 pt-3 border-top">
                <small class="text-muted">
                    <i class="fas fa-lightbulb me-1"></i>
                    Tip: Invalid emails and duplicates will be automatically skipped
                </small>
            </div>
        </div>
    </div>
</div>

<script>
function displayFileName() {
    const input = document.getElementById('csvfile');
    const fileNameDisplay = document.getElementById('fileName');
    if (input.files.length > 0) {
        fileNameDisplay.textContent = '📄 ' + input.files[0].name;
        fileNameDisplay.style.fontWeight = 'bold';
        fileNameDisplay.style.color = 'var(--primary-color)';
    }
}

// Drag and drop functionality
const uploadArea = document.querySelector('.upload-area');

['dragenter', 'dragover', 'dragleave', 'drop'].forEach(eventName => {
    uploadArea.addEventListener(eventName, preventDefaults, false);
});

function preventDefaults(e) {
    e.preventDefault();
    e.stopPropagation();
}

['dragenter', 'dragover'].forEach(eventName => {
    uploadArea.addEventListener(eventName, highlight, false);
});

['dragleave', 'drop'].forEach(eventName => {
    uploadArea.addEventListener(eventName, unhighlight, false);
});

function highlight(e) {
    uploadArea.style.borderColor = 'var(--primary-color)';
    uploadArea.style.background = 'rgba(79, 70, 229, 0.1)';
}

function unhighlight(e) {
    uploadArea.style.borderColor = 'var(--border-color)';
    uploadArea.style.background = 'var(--light-bg)';
}

uploadArea.addEventListener('drop', handleDrop, false);

function handleDrop(e) {
    const dt = e.dataTransfer;
    const files = dt.files;
    document.getElementById('csvfile').files = files;
    displayFileName();
}
</script>

<?php include 'includes/footer.php'; ?>
